import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;

import javax.crypto.Cipher;
import java.io.ByteArrayOutputStream;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.*;

public class RSAUtils {

    public static final String CHARSET = "UTF-8";
    public static final String RSA_ALGORITHM = "RSA";


    public static Map<String, String> createKeys(int keySize){
        KeyPairGenerator kpg;
        try{
            kpg = KeyPairGenerator.getInstance(RSA_ALGORITHM);
        }catch(NoSuchAlgorithmException e){
            throw new IllegalArgumentException("No such algorithm-->[" + RSA_ALGORITHM + "]");
        }

        //初始化KeyPairGenerator對象 密鑰長度
        kpg.initialize(keySize);
        //生產密鑰對
        KeyPair keyPair = kpg.generateKeyPair();
        //得到密鑰
        Key publicKey = keyPair.getPublic();
        //返回一個publicKey經過二次加密后的字符串
        String publicKeyStr = Base64.encodeBase64URLSafeString(publicKey.getEncoded());
        //得到私鑰
        Key privateKey = keyPair.getPrivate();
        //返回一個privateKey經過二次加密后的字符串
        String privateKeyStr = Base64.encodeBase64URLSafeString(privateKey.getEncoded());

        Map<String, String> keyPairMap = new HashMap<String, String>();
        keyPairMap.put("publicKey", publicKeyStr);
        keyPairMap.put("privateKey", privateKeyStr);

        return keyPairMap;
    }

    /**
     * 獲得公鑰
     * @param publicKey 密鑰字符串（經過base64編碼）
     * @throws Exception
     */
    public static RSAPublicKey getPublicKey(String publicKey) throws NoSuchAlgorithmException, InvalidKeySpecException {
        //通過X509編碼的key指令獲得公鑰對象
        KeyFactory keyFactory = KeyFactory.getInstance(RSA_ALGORITHM);
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKey));
        RSAPublicKey key = (RSAPublicKey) keyFactory.generatePublic(x509KeySpec);
        return key;
    }

    /**
     * 獲得私鑰
     * @param privateKey 密鑰字符串（經過base64編碼）
     * @throws Exception
     */
    public static RSAPrivateKey getPrivateKey(String privateKey) throws NoSuchAlgorithmException, InvalidKeySpecException {
        //通過PKCS#8編碼的Key指令獲取私鑰對象
        KeyFactory keyFactory = KeyFactory.getInstance(RSA_ALGORITHM);
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(Base64.decodeBase64(privateKey));
        RSAPrivateKey key = (RSAPrivateKey) keyFactory.generatePrivate(pkcs8KeySpec);
        return key;
    }

    /**
     * 公鑰加密
     * @param data 加密數據
     * @param publicKey 公鑰
     * @return 加密後數據
     */
    public static String publicEncrypt(String data, RSAPublicKey publicKey){
        try{
            Cipher cipher = Cipher.getInstance(RSA_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);
            return Base64.encodeBase64URLSafeString(rsaSplitCodec(cipher, Cipher.ENCRYPT_MODE, data.getBytes(CHARSET), publicKey.getModulus().bitLength()));
        }catch(Exception e){
            throw new RuntimeException("加密字符串[" + data + "]時遇到異常", e);
        }
    }

    /**
     * 私鑰解密
     * @param data 解密數據
     * @param privateKey 私鑰
     * @return 解密後數據
     */

    public static String privateDecrypt(String data, RSAPrivateKey privateKey){
        try{
            Cipher cipher = Cipher.getInstance(RSA_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, privateKey);
            return new String(rsaSplitCodec(cipher, Cipher.DECRYPT_MODE, Base64.decodeBase64(data), privateKey.getModulus().bitLength()), CHARSET);
        }catch(Exception e){
            throw new RuntimeException("解密字符串[" + data + "]時遇到異常", e);
        }
    }

    /**
     * 私鑰加密
     * @param data 加密數據
     * @param privateKey 私鑰
     * @return 加密後數據
     */

    public static String privateEncrypt(String data, RSAPrivateKey privateKey){
        try{
            Cipher cipher = Cipher.getInstance(RSA_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, privateKey);
            return Base64.encodeBase64URLSafeString(rsaSplitCodec(cipher, Cipher.ENCRYPT_MODE, data.getBytes(CHARSET), privateKey.getModulus().bitLength()));
        }catch(Exception e){
            throw new RuntimeException("加密字符串[" + data + "]時遇到異常", e);
        }
    }

    /**
     * 公鑰解密
     * @param data 解密數據
     * @param publicKey 公鑰
     * @return 解密後數據
     */

    public static String publicDecrypt(String data, RSAPublicKey publicKey){
        try{
            Cipher cipher = Cipher.getInstance(RSA_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, publicKey);
            return new String(rsaSplitCodec(cipher, Cipher.DECRYPT_MODE, Base64.decodeBase64(data), publicKey.getModulus().bitLength()), CHARSET);
        }catch(Exception e){
            throw new RuntimeException("解密字符串[" + data + "]時遇到異常", e);
        }
    }

    private static byte[] rsaSplitCodec(Cipher cipher, int opmode, byte[] datas, int keySize){
        int maxBlock = 0;
        if(opmode == Cipher.DECRYPT_MODE){
            maxBlock = keySize / 8;
        }else{
            maxBlock = keySize / 8 - 11;
        }
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int offSet = 0;
        byte[] buff;
        int i = 0;
        try{
            while(datas.length > offSet){
                if(datas.length-offSet > maxBlock){
                    buff = cipher.doFinal(datas, offSet, maxBlock);
                }else{
                    buff = cipher.doFinal(datas, offSet, datas.length-offSet);
                }
                out.write(buff, 0, buff.length);
                i++;
                offSet = i * maxBlock;
            }
        }catch(Exception e){
            throw new RuntimeException("加解密閥值為["+maxBlock+"]的數據時遇到異常", e);
        }
        byte[] resultDatas = out.toByteArray();
        IOUtils.closeQuietly(out);
        return resultDatas;
    }

    //私鑰加密->公鑰解密
    private static void decryptByPublicKey(String publicKey,String rsamsg) throws Exception{
        String requestOrderData = publicDecrypt(rsamsg,getPublicKey(publicKey));
        System.out.println("公鑰解密内容："+URLDecoder.decode(requestOrderData,"UTF-8"));
    }

    //公鑰加密
    private static String encryptByPublicKey(String publicKey) throws Exception{

        String send_time = parseLocalDateTime2String(LocalDateTime.now(),"SSSssmmHHyyyyMMdd");
        String td = "TT" + send_time;

        String web="AAAAAA";
        JSONObject head = new JSONObject(new LinkedHashMap<>());
        head.put("web",web);
        head.put("send_time",send_time);
        JSONObject body = new JSONObject(new LinkedHashMap<>());
        body.put("country_type","cht");
        body.put("mn","200");
        body.put("td",td);
        //公鑰加密内容
        JSONObject beforeJson = new JSONObject(new LinkedHashMap<>());
        beforeJson.put("body",body);
        beforeJson.put("head",head);

        System.out.println(beforeJson.toJSONString());
        System.out.println(URLEncoder.encode(beforeJson.toJSONString(),"UTF-8"));

        return publicEncrypt(URLEncoder.encode(beforeJson.toJSONString(),"UTF-8"),getPublicKey(publicKey));
    }

    //公鑰加密->私鑰解密
    private static void decryptByPrivateKey(String privateKey,String rsamsg) throws Exception{
        String requestOrderData = privateDecrypt(rsamsg,getPrivateKey(privateKey));
        System.out.println("私鑰解密内容："+URLDecoder.decode(requestOrderData,"UTF-8"));
    }

    //私鑰加密
    private static String encryptByPrivateKey(String privateKey) throws Exception{

        String send_time = parseLocalDateTime2String(LocalDateTime.now(),"SSSssmmHHyyyyMMdd");
        String td = "TT" + send_time;
        JSONObject head = new JSONObject(new LinkedHashMap<>());
        head.put("web","ABCDD");
        head.put("send_time",send_time);
        JSONObject body = new JSONObject(new LinkedHashMap<>());
        body.put("card_no","adfadfd");
        body.put("card_type","00");
        body.put("currency","TWD");
        body.put("mn","280");
        body.put("name","A○E");
        body.put("pay_result","20");
        body.put("td",td);
        body.put("trade_no","C"+format(new Date()));
        //公鑰加密内容
        JSONObject jsonObject = new JSONObject(new LinkedHashMap<>());
        jsonObject.put("body",body);
        jsonObject.put("head",head);

        System.out.println(jsonObject.toJSONString());
        System.out.println(URLEncoder.encode(jsonObject.toJSONString(),"UTF-8"));

        return privateEncrypt(URLEncoder.encode(jsonObject.toJSONString(),"UTF-8"),getPrivateKey(privateKey));
    }

    private static String format(Date date) {
        if(date != null){
            SimpleDateFormat df = new SimpleDateFormat("yyyyMMddHHmmss");
            return df.format(date);
        }
        return null;
    }

    private static String parseLocalDateTime2String(LocalDateTime localDateTime, String pattern){
        if(localDateTime == null){
            return "";
        }
        java.time.format.DateTimeFormatter dateTimeFormatter = java.time.format.DateTimeFormatter.ofPattern(pattern, Locale.US);
        return dateTimeFormatter.format(localDateTime);
    }

    public static void main(String[] strings) throws Exception{
        //RSA公鑰
        String publicKey = "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDBDDt3Ba4TXbZ5KYg6oZrtM5xZ1Yv/vk3nfuAI5DTtnIqBFgHGx1BhWTW5+qE4aJad4DG2RzY23qoAMPWQb/OrytOQ6VUJ7ao8FxFGcqBjWzix3znM0rLA/8seMDhv+TJ2bRlXEksGHxY5KAPOHkg8qmdwJjW+76fWhD+WVGsfVwIDAQAB";
        //RSA私鑰
        String privateKey = "MIICeAIBADANBgkqhkiG9w0BAQEFAASCAmIwggJeAgEAAoGBAMEMO3cFrhNdtnkpiDqhmu0znFnVi/++Ted+4AjkNO2cioEWAcbHUGFZNbn6oTholp3gMbZHNjbeqgAw9ZBv86vK05DpVQntqjwXEUZyoGNbOLHfOczSssD/yx4wOG/5MnZtGVcSSwYfFjkoA84eSDyqZ3AmNb7vp9aEP5ZUax9XAgMBAAECgYAwVDai4nolAQYIqTwgqYqWraAL4WlOGj94oAvIThFuFw2ZqX/CWu2loWbQ64znCVADsDfh0UwiSnY23AruIQjYXoQg3eFiDwlWsqtApJZZLLNlds1lyYNnYK2kZ6HjoU7RTReBSzu49qJk1FrbMI4jF+nEDxN1fgmEFKRAL8zlUQJBAPO0pl+VoO9VO5zFvUvKfOkvNfFwSo3RgeJJjgxbSVxDx262EpDZrqJceFuj13RazMj29TLkyJu1lF5djUJ4M2kCQQDKyVvBBousMO6AuWxpzd5IxgeC1H/2EM51qc1YxPZBhb5FQpK2YkAfscHRA42yATosig3fJiNxv4Ppnl4TMiS/AkEAqbDS2+FzAtMFRB8HtLtUiIV/+RJoRyFnX+o5h28UwLLNLdCHUubPy2u5vo1x2ynSR+h1SS+2Y31mxuI7FtpdaQJBAJzpWCUxQmx35GUYlWn9UW3wRamXcnhdotPrEykIJyCmM4d2Lv4PYlzj9/wais7dIj2KiZdM9Io0zga8c0DZQRMCQQDTptqCpE6g5l++DjKc8fvKmyxHDTWUkqUSQNSol3DMQuipLCBpYl1vaAcoqA4pSPBvwfSMXeyh/k84dLLuWnj9";
        //公鑰加密
        String publicRsaMsg = encryptByPublicKey(publicKey);
        System.out.println(publicRsaMsg);
        //公鑰加密->私鑰解密
        decryptByPrivateKey(privateKey,publicRsaMsg);
        //私鑰加密
        String privateRsaMsg = encryptByPrivateKey(privateKey);
        System.out.println(privateRsaMsg);
        //私鑰加密->公鑰解密
        decryptByPublicKey(publicKey,privateRsaMsg);
//        decryptByPublicKey(publicKey,rsamsg);
    }

}
