<?php

namespace Opencart\Catalog\Controller\Extension\SunpayPayment\Payment;

use Exception;
use Opencart\Catalog\Model\Checkout\Order;

/**
 * @author HughesWu extend Roger
 * @author jack
 * @author Stally
 */
class SunpayPayment extends \Opencart\System\Engine\Controller
{
    private $rsaPublicKey, $sha2Key;
    private $sunpayTool, $ei;
    private $paymentSubfix = 'payment_sunpay_payment';
    /**
     * 付款畫面
     *
     *
     * @return html
     */
    public function index()
    {
		$this->load->language('extension/sunpay_payment/payment/SunpayMPG');
        $data['button_confirm'] = $this->language->get('button_confirm');

        $this->load->model('checkout/order');
        $this->load->model('account/order');

        $data['action'] = $this->url->link('extension/sunpay_payment/payment/sunpay_payment.sendPaymentInfo');

        // AJAX 訂單送回 Sunpay 之前執行
        $data['ajaxUrl'] = $this->url->link('extension/sunpay_payment/payment/sunpay_payment.confirm');

        // 要傳遞資料
        $data['params'] = $this->_composePostInput();
        $data['payment_methods'] = $this->_getPaymentMethods();

        $data['inv_status'] = $this->config->get($this->paymentSubfix . '_inv_status');

        $this->cart->clear();

        return $this->load->view('extension/sunpay_payment/payment/SunpayMPG', $data);
    }

    private function initial_sunpayTool(){
        require_once(dirname(__DIR__).'/tools/sunpayTool.php');
        $this->sunpayTool = sunpayTool::get_instance();
    }

    private function initial_sunpayElectronicinvoice(){
        // 發票
        require_once(dirname(__DIR__).'/tools/sunpayElectronicinvoice.php');
        $eiEnable        = $this->config->get($this->paymentSubfix . '_inv_status');
        $eiCompanyID     = trim($this->config->get($this->paymentSubfix . '_inv_company_id'));
        $eiMerchantID    = trim($this->config->get($this->paymentSubfix . '_inv_merchant_id'));
        $eiHashKey       = trim($this->config->get($this->paymentSubfix . '_inv_hash_key'));
        $eiHashIV        = trim($this->config->get($this->paymentSubfix . '_inv_hash_iv'));
        $eiIsSendMessage = trim($this->config->get($this->paymentSubfix . '_inv_is_send_message'));
        $eiIsSendPaper   = trim($this->config->get($this->paymentSubfix . '_inv_is_send_page'));
        $eiTaxType       = trim($this->config->get($this->paymentSubfix . '_inv_tax_type'));
        $eiTaxRate       = trim($this->config->get($this->paymentSubfix . '_inv_tax_rate'));
        $TestMode        = $this->config->get($this->paymentSubfix . '_test_mode');

        $eiData = array(
            'eiEnable'      => $eiEnable,
            'companyID'     => $eiCompanyID,
            'merchantID'    => $eiMerchantID,
            'HashKey'       => $eiHashKey,
            'HashIV'        => $eiHashIV,
            'IsSendMessage' => $eiIsSendMessage,
            'IsSendPaper'   => $eiIsSendPaper,
            'taxType'       => $eiTaxType,
            'taxRate'       => $eiTaxRate,
            'testMode'      => $TestMode
        );
        $this->ei = sunpayElectronicInvoice::get_instance($eiData);
    }

    /**
     * 要傳遞資料
     * @return array 一維陣列 key => Input Post Name, value => Input Post Value
     */
    protected function _composePostInput()
    {

        // 訂購   資料
        $order_info = $this->model_checkout_order->getOrder($this->session->data['order_id']);
        $products = $this->model_account_order->getProducts($this->session->data['order_id']);
        // 訂購人 資料
        $user_info = $this->_getUserInfo();

        $body = array(
            "card_type" => 'BuySafe',//$payment_method,
            "country_type" => "cht",
            "currency" => "TWD",
            "email" => $user_info['Email'],
            "mn"  => intval($order_info['total']),
            "order_info" => $this->getItemDescByProduct($products),
            "sdt" => $user_info['Phone'],
            "sna" => $user_info['LastName'] . $user_info['FirstName'],
            "td" => $this->session->data['order_id'] . 'T' . time(),
            "lgs_flag" => 0,
        );

        return $body;
    }

    /**
     * 取得付款方式
     * 
     * @return array
     */
    protected function _getPaymentMethods(){
        //取得付款方式
        $paytype = trim($this->config->get($this->paymentSubfix . '_paytype'));
        $paytype = json_decode($paytype, true);
        $payment_methods =  array();
        foreach ($paytype as $row) {
            $payment_type = str_replace($this->paymentSubfix . '_method_', '', $row['name']);

            if($payment_type == "SunShipWithPay"){
                //超商取貨不付款另外選擇
                continue;
            }
            if ($row['val'] == 'on') {
                $payment_methods[$payment_type] = $this->language->get($payment_type);
            }
        }

        return $payment_methods;
    }

    /**
     * 訂購人資料
     *
     * @return array 一維陣列 key => 欄位名稱, value => 值
     */
    protected function _getUserInfo()
    {

        if ($this->customer->isLogged()) {
            // account = guest
            // guest = array('id', 'username', ..)
            $result = array(
                'FirstName' => $this->customer->getFirstName() ? $this->customer->getFirstName() : $this->session->data['customer']['firstname'],
                'LastName' => $this->customer->getLastName() ? $this->customer->getLastName() : $this->session->data['customer']['lastname'],
                'Email' => ($this->customer->getEmail() ? $this->customer->getEmail() : $this->session->data['customer']['email']),
                'Phone' => ($this->customer->getTelephone() ? $this->customer->getTelephone() : $this->session->data['customer']['telephone']),
            );
        } elseif (!empty($_POST)) {
            $result = array(
                'FirstName' => $_POST["firstname"],
                'LastName' => $_POST["lastname"],
                'Email' => $_POST["email"],
                'Phone' => $_POST["telephone"],
            );
        } elseif (isset($this->session->data['guest'])) {
            $result = array(
                'FirstName' => isset($this->session->data['guest']['firstname']) ? $this->session->data['guest']['firstname'] : $this->session->data['guest']['payment']['firstname'],
                'LastName' => isset($this->session->data['guest']['lastname']) ? $this->session->data['guest']['lastname'] : $this->session->data['guest']['payment']['lastname'],
                'Email' => isset($this->session->data['guest']['email']) ? $this->session->data['guest']['email'] : $this->session->data['guest']['payment']['email'],
                'Phone' => isset($this->session->data['guest']['telephone']) ? $this->session->data['guest']['telephone'] : $this->session->data['guest']['payment']['telephone'],
            );
        } else {
            $result = array(
                'FirstName' => '',
                'LastName' => '',
                'Email' => '',
                'Phone' => '',
            );
        }

        $result['ShowLanguage'] = $this->language->get('code');
        $result['ShowName'] = strpos($this->language->get('code'), 'zh-TW') !== true ? $result['LastName'] . $result['FirstName'] : $result['FirstName'] . $result['LastName'];

        return $result;
    }

    private function _convert_payment_method_to_code($payment_method){
        $method = array(
            'BuySafe'    => '01',
            'buysafemul' => '01',
            'GooglePayAndApplePay'  => '03',
            'UnionPay'   => '02',
            'Atm'        => '08',
            '24Pay'      => '06',
            'PayCode'    => '07',
            'JkoPay'     => '10',
            'SunShip'    => '09',
        );

        if(isset($method[$payment_method])){
            return $method[$payment_method];
        }else{
            return $payment_method;
        }   
    }

    /**
     * //ready to deprecate
     * convert payment_type
     */
    private function _convert_card_type_name($card_type = '')
    {
        $PaymentType_Ary = array(
            '01' => '信用卡',
            '02' => '銀聯卡',
            '03' => 'ApplePay / GooglePay',
            '06' => '超商代碼繳費',
            '07' => '超商條碼繳費',
            '08' => '虛擬帳號',
            '09' => '超商取貨付款',
            '10' => '街口支付',
        );
        $re_str          = (isset($PaymentType_Ary[$card_type])) ? $PaymentType_Ary[$card_type] : $card_type;
        return $re_str;
    }

    /**
     * 
     * @param string $text 輸入字串
     * @return string 處理後的安全字串
     */
    private function esc_attr($text) {
        // 確保輸入是字串
        $safe_text = (string) $text;

        // 移除不必要的控制字元（避免非可見字元注入）
        $safe_text = preg_replace('/[\x00-\x1F\x7F]/u', '', $safe_text);

        // 使用 htmlspecialchars 轉換 HTML 特殊符號
        // ENT_QUOTES: 將單引號、雙引號都轉換
        // ENT_SUBSTITUTE: 遇到無效的 UTF-8 會用 � 取代
        // 'UTF-8': 預設使用 UTF-8 編碼
        $safe_text = htmlspecialchars($safe_text, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8', false);

        return $safe_text;
    }

    /**
     * 安全地清理文字輸入
     *
     * @param string $str 原始輸入
     * @return string 清理後的文字
     */
    function sanitize_text_field($str) {
        // 1. 確保是字串
        $filtered = (string) $str;

        // 2. 移除 HTML 標籤
        $filtered = strip_tags($filtered);

        // 3. 解碼 HTML 實體，例如 &amp; → &
        $filtered = html_entity_decode($filtered, ENT_QUOTES, 'UTF-8');

        // 4. 移除回車符號、換行符號（保留空白）
        $filtered = preg_replace('/[\r\n\t]+/', ' ', $filtered);

        // 5. 移除開頭與結尾多餘空白
        $filtered = trim($filtered);

        // 6. 移除不合法的控制字元（ASCII 0–31）
        $filtered = preg_replace('/[\x00-\x1F\x7F]/u', '', $filtered);

        // 7. 若含多重空白，合併為單一空白
        $filtered = preg_replace('/\s+/', ' ', $filtered);

        return $filtered;
    }

    /**
     * AJAX 訂單送回 Sunpay 之前執行
     */
    public function confirm()
    {
        $this->load->model('checkout/order');

        $inv_status = $this->config->get($this->paymentSubfix . '_inv_status');

        if ($inv_status == 1) {
            $invData = $_POST;
            //寫入發票資訊
            if ($_POST['billing_needUBN']) {
                $invData['invUBNNum'] = $_POST['billing_UBN'];
                $invData['invBuyer'] = $_POST['billing_Buyer'];
            }

            $invData['carrier_type'] = $_POST['billing_carrierType'];
            $invData['carrier_num'] = $_POST['billing_carrierId1'];

            //inv table
            $this->addInvoiceData($this->session->data['order_id'], $invData);
        }

        $orderModel = new Order($this->registry);

        $orderModel->addHistory(
            $this->session->data['order_id'],
            $this->config->get($this->paymentSubfix . '_order_status_id')
        );
    }

    /**
     * 處理金流資料,成功後跳轉到紅陽付款頁面
     */
    public function sendPaymentInfo(){
        
        $isTestMode = $this->config->get($this->paymentSubfix . '_test_mode');
        $apiUrl = $isTestMode ? 'https://testtrade.sunpay.com.tw/v4/cash' : 'https://trade.sunpay.com.tw/v4/cash';
        //處理金流資料
        /**
         * MPG 各項設定參數
         */
        $this->initial_sunpayTool();
        $this->rsaPublicKey = trim($this->config->get($this->paymentSubfix . '_rsa_key'));
        $this->sha2Key = trim($this->config->get($this->paymentSubfix . '_sha2_key'));
        //取得send_time
        $sendTime = $this->sunpayTool->get_send_time();
        
        $merchant_id = trim($this->config->get($this->paymentSubfix . '_merchant_id'));
        $head = array(
            "send_time" => $sendTime,
            "web" => $merchant_id,
        );

        //取得付款方式
        $payment_info = $_POST;
        $payment_type = $payment_info['card_type'];
        $payment_method = $this->_convert_payment_method_to_code($payment_type);

        $is_use_sunship = false;
        if($payment_info['lgs_flag'] == "1"){
            $is_use_sunship = true;
        }

        $body = array(
            "card_type" => $payment_method,
            "country_type" => $payment_info['country_type'],
            "currency" => $payment_info['currency'],
            "email" => $payment_info['email'],
            "mn"  => strval($payment_info['mn']),
            "order_info" => $payment_info['order_info'],
            "sdt" => $payment_info['sdt'],
            "sna" => $payment_info['sna'],
            "td" => $payment_info['td'],
        );
        //判斷是否有填電話
        // if(!empty($payment_info['sdt'])){
        //     $body['sdt'] = $payment_info['sdt'];
        // }
        //分期付款必要欄位
        if($payment_method == "01" && $payment_type == "buysafemul"){
            $body["term"] = "3";
        }
        //超商條碼/超商代碼選用欄位
        if($payment_method == "06" || $payment_method == "07"){
            // $body["bill_date"] = date("Ymd");
            // $body["due_date"] = "7";
            // $product_data = array();
            // $products = $order->get_items();
            // foreach($products AS $product){
            //     $product_data[] = array(
            //         "no" => $product->get_product_id(),
            //         "product_name" => $product->get_name(),
            //         "product_price" => round($product->get_subtotal() / $product->get_quantity()),
            //         "product_quantity" => $product->get_quantity(),
            //     );
            // }
            // $body["product"] = $product_data;
        }
        //虛擬帳號選用欄位
        if($payment_method == "08"){
            // $body["due_date"] = "7";
        }
        //超商取貨付款
        if($payment_method == "09"){
            // $body["receiver_name"] = $order->get_billing_last_name() . $order->get_billing_first_name();
            // $body["receiver_phone"] = $order->get_billing_phone();
            // $body["receiver_mail"] = $order->get_billing_email();
            $body["lgs_flag"] = "1";
        }
        // ksort($body);
        if($is_use_sunship){
            $body['lgs_flag'] = "1";
        }

        $data = array(
            "body" => $body,
            "head" => $head,
        );

        $encrypted_data = $this->sunpayTool->get_encrypted_data($data, $this->rsaPublicKey, $this->sha2Key);
        $rsamsg    = $encrypted_data['rsamsg'];
        $check_value = $encrypted_data['check_value'];

        $post_data = array(
            "web" => $merchant_id,
            "send_time" => $sendTime,
            "rsamsg" => $rsamsg,
            "check_value" => $check_value
        );

        $sunpay_args_array = array();
        foreach ($post_data as $key => $value) {
            $sunpay_args_array[] = '<input type="hidden" name="' . $this->esc_attr($key) . '" value="' . $this->esc_attr($value) . '" />';
        }

        echo '<form id="sunpay" name="sunpay" action="' . $apiUrl . '" method="post" target="_top">' . implode('', $sunpay_args_array) . '
            <input type="submit" class="button-alt" id="submit_sunpay_payment_form" value="" />
            </form>' . "<script>setTimeout(\"document.forms['sunpay'].submit();\",\"100\")</script>";
    }

    /**
     * PAGE 幕前顯示(客戶端回傳)
     */
    public function feedbackShow()
    {
        /**
         * 頁面資料
         */
        $this->language->load('checkout/success');
        $this->language->load('checkout/success_error');
		$this->load->language('extension/sunpay_payment/payment/SunpayMPG');

        $req_data = array();
        if(!isset($_REQUEST['rsamsg'])){
            // $url = $this->url->link('common/home', '', 'SSL');
            $url = $this->load->link('common/success', 'language=' . $this->config->get('config_language'));
            header("location:$url");
        }

        // 檢查SHA值是否正確 MPG1.4版
        $this->initial_sunpayTool();
        $this->rsaPublicKey = trim($this->config->get($this->paymentSubfix . '_rsa_key'));
        $this->sha2Key = trim($this->config->get($this->paymentSubfix . '_sha2_key'));
        $merchant_id = trim($this->config->get($this->paymentSubfix . '_merchant_id'));
        if (!empty($this->sanitize_text_field($_REQUEST['rsamsg']))) {
            $req_data = $this->sunpayTool->get_decrypted_data(
                $this->sanitize_text_field($_REQUEST['rsamsg']),
                $this->rsaPublicKey,
                $this->sha2Key
            );
            if (!$this->sunpayTool->chkShaIsVaildByReturnData($_REQUEST, $req_data['check_value'], $merchant_id)) {
                // echo '檢查碼錯誤';
                $url = $this->url->link('common/home', '', 'SSL');
                header("location:$url");
                exit;
            }
            if (!is_array($req_data) || 
            (isset($req_data['data']) && !is_array($req_data['data']))) {
                // echo '解密失敗';
                $url = $this->url->link('common/home', '', 'SSL');
                header("location:$url");
                exit;
            }
        }

        // 初始化$req_data 避免因index不存在導致NOTICE 若無傳入值的index將設為null
        $init_indexes_head  = 'send_time,web';
        $head = $this->sunpayTool->init_array_data($req_data['data']['head'],$init_indexes_head);
        $init_indexes_body  = 'approve_code,card_no,card_type,currency,mn,name,note1,note2,pay_agency,pay_agency_memo,
                            pay_date,pay_result,pay_time,td,trade_no,save_card_token_result,user_no';                                                     //付款完成之回傳參數欄位
        $init_indexes_body .= 'delivery_type,store_id,store_name,store_type,lgs_type,receiver_name,receiver_phone';                                          //超商物流之回傳參數欄位
        $init_indexes_body .= 'account_id,account_name,bank_code,banc_name,due_date,pay_code,barcodeA,barcodeB,barcodeC';                                    //取號完成之回傳參數欄位
        $body = $this->sunpayTool->init_array_data($req_data['data']['body'],$init_indexes_body);
        $req_data = array(
            "body" => $body,
            "head" => $head
        );

        $re_Status  = trim($req_data['body']['pay_result']) != '' ? trim($req_data['body']['pay_result']) : null;
        $re_CardType = $req_data['body']['card_type'];

        // 檢查回傳狀態是否為成功
        $msg = "";
        $is_success = true;
        if ($re_Status != "10") {
            $msg = '訂單處理失敗: ';
            if($re_Status == "12" && $re_CardType == "09"){
                //超商取貨另外處理
                $msg = "訂單已建立,等待付款";
            }else{
                if($re_Status == "12" && str_contains("06,07,08",$re_CardType)){
                    //非即時性付款
                    $msg = "訂單已建立,等待付款";
                }else if($re_Status == "11"){
                    $msg .= "交易失敗";
                    $is_success = false;
                }else if($re_Status == "12"){
                    $msg .= "訂單已建立過";
                    $is_success = false;
                }
            }
        }

        // 顯示的 Title
        $title = $is_success ? $this->language->get('heading_title_succ') : $this->language->get('heading_title_fail') . $msg;

        $this->document->setTitle($title);

        $data['breadcrumbs'] = array();

        $data['breadcrumbs'][] = array(
            'href' => $this->url->link('common/home'),
            'text' => $this->language->get('text_home'),
            'separator' => false
        );

        $data['breadcrumbs'][] = array(
            'href' => $this->url->link('checkout/cart'),
            'text' => $this->language->get('text_basket'),
            'separator' => $this->language->get('text_separator')
        );

        $data['breadcrumbs'][] = array(
            'href' => $this->url->link('checkout/checkout', '', 'SSL'),
            'text' => $this->language->get('text_checkout'),
            'separator' => $this->language->get('text_separator')
        );

        $data['breadcrumbs'][] = array(
            'href' => $this->url->link('checkout/success'),
            'text' => $this->language->get('text_success'),
            'separator' => $this->language->get('text_separator')
        );

        $data['heading_title'] = $title;

        if ($this->customer->isLogged()) {
            // 顯示的結果
            $show_text_customer = $is_success ? $this->language->get('text_customer') : $this->language->get('text_customer_fail');

            $data['text_message'] = sprintf($show_text_customer, $this->url->link('account/account', '', 'SSL'), $this->url->link('account/order', '', 'SSL'), $this->url->link('account/download', '', 'SSL'), $this->url->link('information/contact'));
        } else {

            // 顯示的結果
            $show_text_guest = $is_success ? $this->language->get('text_guest') : $this->language->get('text_guest_fail');

            $data['text_message'] = sprintf($show_text_guest, $this->url->link('information/contact'));
        }

        $data['button_continue'] = $this->language->get('button_continue');

        $data['continue'] = $this->url->link('common/home');

        //訂單成功後，購物車清空
        if ($is_success) {
            $this->clearCustomerCart($this->customer->getId());
        }

        $data['column_left'] = $this->load->controller('common/column_left');
        $data['column_right'] = $this->load->controller('common/column_right');
        $data['content_top'] = $this->load->controller('common/content_top');
        $data['content_bottom'] = $this->load->controller('common/content_bottom');
        $data['footer'] = $this->load->controller('common/footer');
        $data['header'] = $this->load->controller('common/header');

        // $this->feedback(); // renew payment status, in case of user close notify url

        $this->response->setOutput($this->load->view('common/success', $data));
    }

    /**
     * POST 幕後更新主機端回傳
     */
    public function feedback()
    {
        try{
            $req_data = array();
            if(!isset($_REQUEST['rsamsg'])){
                return false;
            }

            //初始化資料
            $this->load->model('checkout/order');
            $this->initial_sunpayTool();
            $this->rsaPublicKey = trim($this->config->get($this->paymentSubfix . '_rsa_key'));
            $this->sha2Key = trim($this->config->get($this->paymentSubfix . '_sha2_key'));
            $merchant_id = trim($this->config->get($this->paymentSubfix . '_merchant_id'));

            // 檢查SHA值是否正確 MPG1.4版
            if (!empty($this->sanitize_text_field($_REQUEST['rsamsg']))) {
                $req_data = $this->sunpayTool->get_decrypted_data(
                    $this->sanitize_text_field($_REQUEST['rsamsg']),
                    $this->rsaPublicKey,
                    $this->sha2Key
                );
                if (!$this->sunpayTool->chkShaIsVaildByReturnData($_REQUEST, $req_data['check_value'], $merchant_id)) {
                    // $this->writeLog('檢查碼錯誤');
                    throw new Exception("檢查碼錯誤");
                }
                if (!is_array($req_data) || 
                (isset($req_data['data']) && !is_array($req_data['data']))) {
                    // $this->writeLog('解密失敗');
                    throw new Exception("解密失敗");
                }
            }

            // 初始化$req_data 避免因index不存在導致NOTICE 若無傳入值的index將設為null
            $init_indexes_head  = 'send_time,web';
            $head = $this->sunpayTool->init_array_data($req_data['data']['head'],$init_indexes_head);
            $init_indexes_body  = 'approve_code,card_no,card_type,currency,mn,name,note1,note2,pay_agency,pay_agency_memo,
                                pay_date,pay_result,pay_time,td,trade_no,save_card_token_result,user_no';                                                     //付款完成之回傳參數欄位
            $init_indexes_body .= 'delivery_type,store_id,store_name,store_type,lgs_type,receiver_name,receiver_phone';                                          //超商物流之回傳參數欄位
            $init_indexes_body .= 'account_id,account_name,bank_code,banc_name,due_date,pay_code,barcodeA,barcodeB,barcodeC';                                    //取號完成之回傳參數欄位
            $body = $this->sunpayTool->init_array_data($req_data['data']['body'],$init_indexes_body);
            $req_data = array(
                "body" => $body,
                "head" => $head
            );

            $re_MerchantOrderNo = trim($req_data['body']['td']);
            $re_Status  = trim($req_data['body']['pay_result']) != '' ? trim($req_data['body']['pay_result']) : null;
            $re_TradeNo = $req_data['body']['trade_no'];
            $re_Amt     = $req_data['body']['mn'];
            $re_CardType = $req_data['body']['card_type'];

            // 取得該筆交易資料
            $order_id = explode('T', $re_MerchantOrderNo)[0];

            $order_info = $this->model_checkout_order->getOrder($order_id);
            
            // 是否有資料
            if (empty($order_info)) {
                throw new Exception($re_MerchantOrderNo . ": DataError");
            }

            /**
             *  取得 付費方式 相關資料
             */
            $this->load->model('setting/setting');

            $store_info = $this->model_setting_setting->getSetting(
                'payment_sunpay_payment',
                $order_info['store_id']
            );

            // 檢查回傳狀態是否為成功
            if ($re_Status != "10") {
                $msg = '訂單處理失敗: ';
                if($re_Status == "12" && $re_CardType == "09"){
                    //超商取貨另外處理
                    $msg = "付款方式:" . $this->_convert_card_type_name($re_CardType);
                    $msg .= "，訂單已建立,等待付款";
                    $this->updateOrderComment(
                        $order_info['order_id'],
                        $msg
                    );
                }else{
                    if($re_Status == "12" && str_contains("06,07,08",$re_CardType)){
                        //非即時性付款
                        $msg = "付款方式:" . $this->_convert_card_type_name($re_CardType);
                        $msg .= "，訂單已建立,等待付款";
                        $this->updateOrderComment(
                            $order_info['order_id'],
                            $msg
                        );
                    }else if($re_Status == "11"){
                        $msg .= "交易失敗";
                        $this->updateOrderStatusToFail($order_info['order_id'], $store_info, '錯誤訊息: ' . $msg);
                    }else if($re_Status == "12"){
                        $msg .= "訂單已建立過";
                        $this->updateOrderStatusToFail($order_info['order_id'], $store_info, '錯誤訊息: ' . $msg);
                    }
                    throw new Exception($this->esc_attr($msg));
                }
            }

            //檢查訂單是否已付款
            if ($order_info['order_status_id'] == $store_info[$this->paymentSubfix . '_order_finish_status_id']) {
                throw new Exception('訂單已付款');
            }

            // 檢查交易總金額
            $Amt = intval($order_info['total']);
            if ($Amt != $re_Amt) {
                $content = $re_MerchantOrderNo . ': ERROR_2';
                $this->updateOrderStatusToFail($order_info['order_id'], $store_info, '錯誤訊息: ' . '付款訂單金額不一致');
                throw new Exception($content);
            }

            // 非即時性付款(超商取貨付款另外處理)
            if(str_contains("06,07,08",$re_CardType)){
                try{
                    // 檢查是否付款
                    if (empty($req_data['body']['pay_date']) || empty($req_data['body']['pay_time'])) {
                        $comment = '訂單付款方式屬非即時性付款,尚未付款';
                        $this->updateOrderComment($order_info['order_id'], $comment);
                        throw new Exception($this->esc_attr($comment));
                    }
                }catch(Exception $e){
                    $msg .= '處理'.$this->_convert_card_type_name($re_CardType).'失敗,';
                    $msg .= '錯誤訊息:' . $e->getMessage();
                    $this->updateOrderStatusToFail($order_info['order_id'], $store_info, $msg);
                    throw new Exception($this->esc_attr($msg));
                }            
            }

            // 超商取貨
            if (!empty($req_data['body']['receiver_name']) || !empty($req_data['body']['store_name']) || !empty($req_data['body']['store_id']) || !empty($req_data['body']['store_type'])) {
                $storeType = urldecode($req_data['body']['store_type']); // 門市類型: 1:7-11, 2:全家, 3:OK, 4:萊爾富
                $storeName = urldecode($req_data['body']['store_name']); // 門市名稱
                $storeId = urldecode($req_data['body']['store_id']); // 門市編號
                $name      = urldecode($req_data['body']['receiver_name']); // 取貨人姓名
                $phone     = $req_data['body']['receiver_phone'];
                $storeTypeStr = array(
                    '1' => '7-11',
                    '2' => '全家',
                    '3' => 'OK',
                    '4' => '萊爾富',
                );
                // 訂單備註
                $comment = '取貨超商：' . $storeTypeStr[$storeType];
                $comment .= '，取貨門市：' . $storeName;
                $comment .= '，取貨門市ID：' . $storeId;
                $comment .= '，取貨人姓名：' . $name;
                $comment .= '，取貨人電話：' . $phone;

                $this->updateOrderComment($order_info['order_id'], $comment);
            }
            if (!empty($req_data['body']['pay_date']) && !empty($req_data['body']['pay_time'])) {
                //寫入付款時間
                $paid_date = date("Y/m/d",strtotime($req_data['body']['pay_date'])) . " " . $req_data['body']['pay_time'];
                $comment = '付款成功';
                $comment .= '，付款時間:' . $paid_date;
                $comment .= '，付款方式:' . $this->_convert_card_type_name($re_CardType);

                // 全部確認過後，修改訂單狀態(處理中，並寄通知信)
                $this->updateOrderStatusToSuccess(
                    $order_info['order_id'],
                    $store_info,
                    $comment,
                );
                $msg   = '訂單修改成功';

                //若有啟用功能則開立發票
                // if ($this->eiEnable == 'yes') {
                //     $this->ei->electronic_invoice($order, $re_TradeNo);
                // }
            }

            $this->create_electronic_invoice($order_info['order_id'], $re_TradeNo);
            $this->writeLog($re_MerchantOrderNo . ' OK');    //訂單成功

            echo "success";
            return "success";

        } catch (Exception $e) {
            $this->writeLog($e->getMessage());
            return $e->getMessage();
        }
        return true;
    }

    private function create_electronic_invoice($order_id, $trade_no){
        if ($order_id != '') {
			//固定使用台灣時間(UTC+8)
			date_default_timezone_set("Asia/Taipei");

            $this->load->model('checkout/order');

            // check inv status
            $inv_status = $this->config->get($this->paymentSubfix . '_inv_status');

            if ($inv_status == 1) {
         
				// 商品資訊
                $order = $this->model_checkout_order->getOrder($order_id);
                $products          = $this->model_checkout_order->getProducts($order_id);        // 訂購商品
                //取得發票資料
                $inv_data = $this->db->query("SELECT * FROM " . DB_PREFIX . "inv_data WHERE order_id = '" . (int)$order_id . "'");
                $inv_data = $inv_data->row;

                $this->initial_sunpayElectronicinvoice();
                $curl_data = $this->ei->electronic_invoice($order_id, $trade_no, $order, $products, $inv_data);

                $url = $curl_data['url'];
                $post_data_json = $curl_data['post_data_json'];
                $result = $this->_curl( $url, $post_data_json, "POST" ); // 背景送出
                $this->writeLog(json_encode($result), false);

                // Add order notes on admin
                $respondDecode = json_decode( $result['web_info'], true );
                if ( $respondDecode['status'] == "SUCCESS" ) {
                    $resultDecode   = $respondDecode['result'];
                    $tradeNumber = $resultDecode['tradeNumber'];
                    $invoiceNumber  = $resultDecode['invoiceNumber'];
                    $comment      = $respondDecode['message'] . '<br>紅陽開立序號: ' . $tradeNumber . '<br>' . '發票號碼: ' . $invoiceNumber;
                    if(isset($inv_data['INV_UBN']) && !empty($inv_data['INV_UBN'])){
                        $comment .= "<br>統一編號: " .  $inv_data['INV_UBN'];
                    }
                    if(isset($inv_data['INV_UBN_BUYER']) && !empty($inv_data['INV_UBN_BUYER'])){
                        $comment .= "<br>買受人名稱: " .  $inv_data['INV_UBN_BUYER'];
                    }
                    if(isset($inv_data['CARRIER_TYPE']) && !empty($inv_data['CARRIER_TYPE'])){
                        switch($inv_data['CARRIER_TYPE']){
                            case 1:
                                $comment .= "<br>手機條碼載具: " .  $inv_data['CARRIER_NUM'];
                                break;
                            case 2:
                                $comment .= "<br>自然人憑證條碼載具: " .  $inv_data['CARRIER_NUM'];
                                break;
                            case 4:
                                $comment .= "<br>捐贈碼: " .  $inv_data['CARRIER_NUM'];
                                break;
                            default:
                                break;
                        }
                    }
                } else {
                    $comment = '發票開立失敗<br>錯誤訊息：' . $respondDecode['errors'][0];
                }
                // $this->writeLog()
                $this->updateOrderComment($order_id, $comment);

            }
        }
    }

	function _curl($url = '', $parameter = '', $post_type = '')
    {
        $curl_options = array(
            CURLOPT_URL => $url,
            CURLOPT_HEADER => false,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_USERAGENT => $this->request->server['HTTP_USER_AGENT'],
            CURLOPT_SSL_VERIFYPEER => TRUE,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,
        );

        if ($post_type == 'POST') {
            $curl_options[CURLOPT_POST] = "1";
            $curl_options[CURLOPT_HTTPHEADER] = [
                'Content-Type: application/json',
                'Accept: application/json'
            ];
            $curl_options[CURLOPT_POSTFIELDS] = $parameter;
        }

        $ch = curl_init();
        curl_setopt_array($ch, $curl_options);
        $result = curl_exec($ch);
        $retcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_errno($ch);

        curl_close($ch);

        $return_info = array(
            "url" => $url,
            "sent_parameter" => $parameter,
            "http_status" => $retcode,
            "curl_error_no" => $curl_error,
            "web_info" => $result,
        );

        return $return_info;
    }

    /**
     * 修改訂單至失敗狀態
     *
     * @param string $order_id 訂單編號
     * @param array $store_info  商店資訊
     * @param string $comment 訂單備註
     */
    private function updateOrderStatusToFail($order_id, $store_info, $comment = '')
    {
        if (!isset($order_id)) return false;
        if (!isset($store_info)) return false;
        if (is_array($order_id)) return false;
        $this->db->query("UPDATE `" . DB_PREFIX . "order` SET order_status_id = '" . (int) $store_info[$this->paymentSubfix . '_order_fail_status_id'] . "', date_modified = NOW() WHERE order_id = '" . $order_id . "'");
        $this->db->query("INSERT INTO " . DB_PREFIX . "order_history SET order_id = '" . $order_id . "', order_status_id = '" . (int) $store_info[$this->paymentSubfix . '_order_fail_status_id'] . "', notify = '1', comment = '" . $this->db->escape($comment) . "', date_added = NOW()");
    }

    /**
     * 修改訂單至成功狀態
     *
     * @param string $order_id 訂單編號
     * @param array $store_info  商店資訊
     * @param string $comment 訂單備註
     */
    private function updateOrderStatusToSuccess($order_id, $store_info, $comment = '')
    {
        if (!isset($order_id)) return false;
        if (!isset($store_info)) return false;
        if (is_array($order_id)) return false;
        $this->db->query("UPDATE `" . DB_PREFIX . "order` SET order_status_id = '" . (int) $store_info[$this->paymentSubfix . '_order_finish_status_id'] . "', comment = '" . $this->db->escape($comment) . "', date_modified = NOW() WHERE order_id = '" . $order_id . "'");
        $this->db->query("INSERT INTO " . DB_PREFIX . "order_history SET order_id = '" . $order_id . "', order_status_id = '" . (int) $store_info[$this->paymentSubfix . '_order_finish_status_id'] . "', notify = '1', comment = '" . $this->db->escape($comment) . "', date_added = NOW()");
    }

    /**
     * 僅修改訂單備註
     *
     * @param string $order_id 訂單編號
     * @param array $store_info  商店資訊
     * @param string $comment 訂單備註
     */
    private function updateOrderComment($order_id, $comment = '')
    {
        if (!isset($order_id)) return false;
        if (is_array($order_id)) return false;
        // $this->db->query("UPDATE `" . DB_PREFIX . "order` SET comment = '" . $this->db->escape($comment) . "', date_modified = NOW() WHERE order_id = '" . $order_id . "'");
        $this->db->query("INSERT INTO " . DB_PREFIX . "order_history SET order_id = '" . $order_id . "', notify = '1', comment = '" . $this->db->escape($comment) . "', date_added = NOW()");
    }

    /**
     * 新增發票資料
     * 
     * @param string $order_id 訂單編號
     * @param array $inv_data 發票資料
     */
    private function addInvoiceData($order_id, $inv_data)
    {
        $INV_UBN = isset($inv_data['invUBNNum']) ? $inv_data['invUBNNum'] : "";
        $INV_UBN_BUYER = isset($inv_data['invBuyer']) ? $inv_data['invBuyer'] : "";
        $INV_CARRIER_TYPE = $inv_data['carrier_type'];
        $INV_CARRIER_NUM = $inv_data['carrier_num'];

        $this->db->query(
            "INSERT INTO " .
                DB_PREFIX . "inv_data " .
                "SET ORDER_ID = '" . $this->db->escape($order_id) . "',
                INV_UBN = '" . $this->db->escape($INV_UBN) . "',
                INV_UBN_BUYER = '" . $this->db->escape($INV_UBN_BUYER) . "',
                CARRIER_TYPE = '" . $this->db->escape($INV_CARRIER_TYPE) . "',
                CARRIER_NUM = '" . $this->db->escape($INV_CARRIER_NUM) . "',
                CREATETIME = NOW();
            "
        );
    }

    public function clearCustomerCart($customer_id)
    {
        // $this->db->query("UPDATE `" . DB_PREFIX . "customer` SET cart = '' WHERE customer_id = '" . (int) $customer_id . "'");
        unset($this->session->data['customer_id']);
    }

    private function getItemDescByProduct($products)
    {
        if (!isset($products)) return $this->config->get($this->paymentSubfix . '_item_desc');
        $item_desc = '';
        foreach ($products as $_ind => $product) {
            if (($_ind + 1) == count($products)) {
                $item_desc .= $product['name'] . ' x ' . $product['quantity'];
                break;
            }
            $item_desc .= $product['name'] . ' x ' . $product['quantity'] . ', ';
        }
        if (strlen($item_desc) > 50) return $this->config->get($this->paymentSubfix . '_item_desc');
        return $item_desc;
    }

    private function writeLog($msg = '', $with_input = true)
    {
        $file_path = DIR_LOGS; // 檔案路徑
        if (!is_dir($file_path)) {
            return;
        }

        $file_name = 'sunpay_' . date('Ymd', time()) . '.txt';  // 取時間做檔名 (YYYYMMDD)
        $file = $file_path . $file_name;
        $fp = fopen($file, 'a');
        $input = ($with_input) ? '|REQUEST:' . json_encode($_REQUEST) : '';
        $log_str = date('Y-m-d h:i:s') . '|' . $msg . $input . "\n\n";
        fwrite($fp, $log_str);
        fclose($fp);
        $this->clean_old_log($file_path);
    }

    private function clean_old_log($dir)
    {
        $del_date = date('Ymd', strtotime('-30 day'));
        $scan_dir = glob($dir . 'sunpay_*.txt');
        foreach ($scan_dir as $value) {
            $date = explode('_', basename($value, '.txt'));
            if (strtotime($del_date) > strtotime($date[1])) {
                unlink($value);
            }
        }
    }
}
