<?php

namespace Sunpay\SunpayPayment\Helper\Services\Config;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Sales\Model\Order;
use Psr\Log\LoggerInterface ;

use Sunpay\SunpayPayment\Helper\Services\Config\MainService;
use Sunpay\SunpayPayment\Helper\Foundation\EncryptionsHelper;

class PaymentService extends AbstractHelper
{
    /**
     * 不指定付款方式。
     */
    public const ALL = '';

    /**
     * 信用卡付費
     */
    public const CREDIT = 'Buysafe';

    /**
     * 信用卡付費(分期)
     */
    public const CREDITINS = 'buysafemul';

    /**
     * 銀聯卡
     */
    public const UNIONPAY = 'UnionPay';

    /**
     * Google pay And Apple pay
     */
    public const GOOGLEPAYANDAPPLEPAY = 'GooglePayAndApplePay';

    /**
     * 虛擬帳號
     */
    public const ATM = 'Atm';

    /**
     * 超商代碼
     */
    public const _24PAY = '24Pay';

    /**
     * 超商條碼
     */
    public const PAYCODE = 'Paycode';

    /**
     * 街口支付
     */
    public const JKOPAY = 'JkoPay';

    /**
     * 超商取貨付款
     */
    public const SUNSHIP = 'SunShip';

    /**
     * 付款成功代碼
     */
    public const PAYMENT_SUCCESS_CODE = 10;

    /**
     * @var LoggerInterface
     */
    protected $_loggerInterface;

    /**
     * @var MainService
     */
    protected $_mainService;

    /**
     * @var EncryptionsHelper
     */
    protected $_encryptionsHelper;

    /**
     * @param Context $context
     */
    public function __construct(
        Context $context,

        LoggerInterface $loggerInterface,
        MainService $mainService,
        EncryptionsHelper $encryptionsHelper
    )
    {
        $this->_loggerInterface = $loggerInterface;
        $this->_mainService = $mainService;
        $this->_encryptionsHelper = $encryptionsHelper;

        parent::__construct($context);
    }

    public function setCheckoutSession(){
        
    }

    /**
     * 前往 AIO
     *
     * @param  array  $accountInfo
     * @param  array  $input
     * @param  string $apiUrl
     * @return string
     */
    public function checkout(array $accountInfo, array $input, string $apiUrl)
    {
        $input = $this->checkoutPrepare($accountInfo, $input);

        $form = $this->generate_sunpay_form($input, $apiUrl);
        return $form;
    }

    /**
     * Generate the sunpay button link (POST method)
     *
     * @access public
     * @param mixed $order_id
     * @return string
     */
    public function generate_sunpay_form($sunpay_args, $sunpay_gateway)
    {
        foreach ($sunpay_args as $key => $value) {
            $sunpay_args_array[] = '<input type="hidden" name="' . $this->esc_attr($key) . '" value="' . $this->esc_attr($value) . '" />';
        }

        return '<form id="sunpay" name="sunpay" action="' . $sunpay_gateway . '" method="post" target="_top">' . implode('', $sunpay_args_array) . '
            <input type="submit" class="button-alt" id="submit_sunpay_payment_form" value="前往 紅陽金流 支付頁面" />
            </form>' . "<script>setTimeout(\"document.forms['sunpay'].submit();\",\"100\")</script>";
    }

    /**
     * 
     * @param string $text 輸入字串
     * @return string 處理後的安全字串
     */
    public function esc_attr($text) {
        // 確保輸入是字串
        $safe_text = (string) $text;

        // 移除不必要的控制字元（避免非可見字元注入）
        $safe_text = preg_replace('/[\x00-\x1F\x7F]/u', '', $safe_text);

        // 使用 htmlspecialchars 轉換 HTML 特殊符號
        // ENT_QUOTES: 將單引號、雙引號都轉換
        // ENT_SUBSTITUTE: 遇到無效的 UTF-8 會用 � 取代
        // 'UTF-8': 預設使用 UTF-8 編碼
        $safe_text = htmlspecialchars($safe_text, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8', false);

        return $safe_text;
    }

    /**
     * 安全地清理文字輸入
     *
     * @param string $str 原始輸入
     * @return string 清理後的文字
     */
    function sanitize_text_field($str) {
        // 1. 確保是字串
        $filtered = (string) $str;

        // 2. 移除 HTML 標籤
        $filtered = strip_tags($filtered);

        // 3. 解碼 HTML 實體，例如 &amp; → &
        $filtered = html_entity_decode($filtered, ENT_QUOTES, 'UTF-8');

        // 4. 移除回車符號、換行符號（保留空白）
        $filtered = preg_replace('/[\r\n\t]+/', ' ', $filtered);

        // 5. 移除開頭與結尾多餘空白
        $filtered = trim($filtered);

        // 6. 移除不合法的控制字元（ASCII 0–31）
        $filtered = preg_replace('/[\x00-\x1F\x7F]/u', '', $filtered);

        // 7. 若含多重空白，合併為單一空白
        $filtered = preg_replace('/\s+/', ' ', $filtered);

        return $filtered;
    }

    /**
     * 整理傳送到 AIO 的參數
     *
     * @param  array $input
     * @return array $send
     */
    public function checkoutPrepare(array $accountInfo, array $input)
    {
        //處理金流資料
        /**
         * MPG 各項設定參數
         */
        $head = array(
            "send_time" => $input['send_time'],
            "web" => $accountInfo['MerchantId'],
        );

        //取得付款方式
        $payment_type = $input['paymentMethod'];
        $payment_method = $this->getSunpayPaymentMethodsCode($payment_type);

        $is_use_sunship = false;
        if($input['lgs_flag'] == "1"){
            $is_use_sunship = true;
        }

        $body = array(
            "card_type" => $payment_method,
            "country_type" => $input['country_type'],
            "currency" => $input['currency'],
            "email" => $input['email'],
            "mn"  => strval($input['mn']),
            "order_info" => $input['order_info'],
            "sdt" => $input['sdt'],
            "sna" => $input['sna'],
            "td" => $input['td'],
        );
        //判斷是否有填電話
        // if(!empty($payment_info['sdt'])){
        //     $body['sdt'] = $payment_info['sdt'];
        // }
        //分期付款必要欄位
        if($payment_method == "01" && $payment_type == "buysafemul"){
            $body["term"] = "3";
        }
        //超商條碼/超商代碼選用欄位
        if($payment_method == "06" || $payment_method == "07"){
            // $body["bill_date"] = date("Ymd");
            // $body["due_date"] = "7";
            // $product_data = array();
            // $products = $order->get_items();
            // foreach($products AS $product){
            //     $product_data[] = array(
            //         "no" => $product->get_product_id(),
            //         "product_name" => $product->get_name(),
            //         "product_price" => round($product->get_subtotal() / $product->get_quantity()),
            //         "product_quantity" => $product->get_quantity(),
            //     );
            // }
            // $body["product"] = $product_data;
        }
        //虛擬帳號選用欄位
        if($payment_method == "08"){
            // $body["due_date"] = "7";
        }
        //超商取貨付款
        if($payment_method == "09"){
            // $body["receiver_name"] = $order->get_billing_last_name() . $order->get_billing_first_name();
            // $body["receiver_phone"] = $order->get_billing_phone();
            // $body["receiver_mail"] = $order->get_billing_email();
            $body["lgs_flag"] = "1";
        }
        // ksort($body);
        if($is_use_sunship){
            $body['lgs_flag'] = "1";
        }

        $data = array(
            "body" => $body,
            "head" => $head,
        );

        $encrypted_data = $this->_encryptionsHelper->get_encrypted_data($data, $accountInfo['RSAPublicKey'], $accountInfo['SHA2Key']);
        $rsamsg    = $encrypted_data['rsamsg'];
        $check_value = $encrypted_data['check_value'];

        $post_data = array(
            "web" => $accountInfo['MerchantId'],
            "send_time" => $input['send_time'],
            "rsamsg" => $rsamsg,
            "check_value" => $check_value
        );

        return $post_data;
    }

    /**
     * 轉換訂購商品格式符合金流訂單API
     *
     * @param  array   $orderItem
     * @return string  $itemName
     */
    public function convertToPaymentItemName($orderItem)
    {
        $itemName = '';

        foreach ($orderItem as $key => $value) {

            $itemName .= $value['name'] . '#' ;
        }

        return $itemName;
    }

    /**
     * Get the amount
     *
     * @param  mixed $amount Amount
     * @return integer
     */
    public function getAmount($amount = 0)
    {
        return round($amount, 0);
    }

    /**
     * 取出API介接網址
     *
     * @param  string  $action
     * @param  int     $stage
     * @return string  $url
     */
    public function getApiUrl(string $action = 'check_out', int $stage = 1)
    {

        if ($stage == 1) {

            switch ($action) {

                case 'check_out':
                    $url = 'https://testtrade.sunpay.com.tw/v4/cash' ;
                break;

                default:
                    $url = '' ;
                break;
            }

        } else {

            switch ($action) {

                case 'check_out':
                    $url = 'https://trade.sunpay.com.tw/v4/cash' ;
                break;

                default:
                    $url = '' ;
                break;
            }
        }

        return $url;
    }

    /**
     * 是否已啟用超商取貨不付款
     */
    public function getSunShipWithPaySetting(){
        return $this->_mainService->getPaymentConfig('enabled_sunpay_sunshipwithpay');
    }

    /**
     * 取得已啟用付款方式
     *
     * @return array
     */
    public function getSunpayPaymentMethods()
    {
        $method = array(
            $this::CREDIT    => '信用卡一次付清',
            $this::CREDITINS => '信用卡(分期)',
            $this::GOOGLEPAYANDAPPLEPAY  => 'Google Pay/Apple Pay',
            $this::UNIONPAY   => '銀聯卡',
            $this::ATM        => '虛擬帳號',
            $this::_24PAY      => '超商條碼繳費',
            $this::PAYCODE    => '超商代碼繳費',
            $this::JKOPAY     => '街口支付',
            $this::SUNSHIP    => '超商取貨付款',
        );

        $enabled_methods = array();
        foreach($method AS $key => $text){
            //取得付款方式設定
            $isEnabled = $this->_mainService->getPaymentConfig('enabled_sunpay_'.strtolower($key));
            $this->_loggerInterface->debug('PaymentService enabled_sunpay_'.strtolower($key).': '.$isEnabled);
            if($isEnabled == 1){
                $enabled_methods[$key] = $text;
            }
        }
        $this->_loggerInterface->debug('PaymentService enabled_methods: '.print_r($enabled_methods, true));

        return $enabled_methods;
    }

    public function getSunpayPaymentMethodsCode($payment_method){
        $method = array(
            $this::CREDIT    => '01',
            $this::CREDITINS => '01',
            $this::GOOGLEPAYANDAPPLEPAY  => '03',
            $this::UNIONPAY   => '02',
            $this::ATM        => '08',
            $this::_24PAY      => '06',
            $this::PAYCODE    => '07',
            $this::JKOPAY     => '10',
            $this::SUNSHIP    => '09',
        );

        if(isset($method[$payment_method])){
            return $method[$payment_method];
        }else{
            return $payment_method;
        }   
    }

    public function getSunpayPaymentMethodsName($payment_method){
        $method = array(
            '01' => $this::CREDIT,
            // '01' => $this::CREDITINS,
            '03' => $this::GOOGLEPAYANDAPPLEPAY,
            '02' => $this::UNIONPAY,
            '08' => $this::ATM,
            '06' => $this::_24PAY,
            '07' => $this::PAYCODE,
            '10' => $this::JKOPAY,
            '09' => $this::SUNSHIP,
        );

        if(isset($method[$payment_method])){
            return $method[$payment_method];
        }else{
            return $payment_method;
        }   
    }

    /**
     * 取得分期期數
     *
     * @return array
     */
    public function getSunpayPaymentMethodNameByCardType($card_type)
    {
        $method = array(
            '01'    => '信用卡一次付清',
            '01-2' => '信用卡(分期)',
            '03'  => 'Google Pay/Apple Pay',
            '02'   => '銀聯卡',
            '08'        => '虛擬帳號',
            '06'      => '超商條碼繳費',
            '07'    => '超商代碼繳費',
            '10'     => '街口支付',
            '09'    => '超商取貨付款',
        );

        if(isset($method[$card_type])){
            return $method[$card_type];
        }else{
            return $card_type;
        }  
    }

    /**
     * 取得紅陽金流
     *
     * @return array
     */
    public function getSunpayAllPayment()
    {
        return [
            $this::CREDIT,
            $this::CREDITINS,
            $this::UNIONPAY,
            $this::GOOGLEPAYANDAPPLEPAY,
            $this::_24PAY,
            $this::PAYCODE,
            $this::SUNSHIP,
            $this::JKOPAY,
        ];
    }

    /**
     * 取得可用的分期期數
     *
     * @return array
     */
    public function getValidCreditInstallments()
    {
        $creditInstallments = $this->_mainService->getPaymentModuleConfig('payment/sunpay_payment_gateway', 'sunpay_payment');

        if (empty($creditInstallments)) {
            return [];
        }

        $trimed = trim($creditInstallments);
        return explode(',', $trimed);
    }

    /**
     * 判斷是否為紅陽金流
     *
     * @param  string $paymentMethod
     * @return bool
     */
    public function isSunpayPayment(string $paymentMethod)
    {
        return in_array($paymentMethod, $this->getSunpayAllPayment());
        // return in_array($paymentMethod, ["sunpay_payment_gateway"]);
    }

    /**
     * 檢查是否為可用的分期期數
     *
     * @param  string $choosenCreditInstallment
     * @return bool
     */
    public function isValidCreditInstallment(string $choosenCreditInstallment)
    {
        $creditInstallments = $this->getValidCreditInstallments();
        return (in_array($choosenCreditInstallment, $creditInstallments));
    }

    /**
     * Filter the inputs
     *
     * @param array $source Source data
     * @param array $whiteList White list
     * @return array
     */
    public function only($source = array(), $whiteList = array())
    {
        $variables = array();

        // Return empty array when do not set white list
        if (empty($whiteList) === true) {
            return $source;
        }

        foreach ($whiteList as $name) {
            if (isset($source[$name]) === true) {
                $variables[$name] = $source[$name];
            } else {
                $variables[$name] = '';
            }
        }
        return $variables;
    }

    /**
     * Validate the amounts
     *
     * @param  mixed $source Source amount
     * @param  mixed $target Target amount
     * @return boolean
     */
    public function validAmount($source = 0, $target = 0)
    {
        return ($this->getAmount($source) === $this->getAmount($target));
    }
}