<?php

namespace Sunpay\SunpayPayment\Helper\Services\Config;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\UrlInterface;
use \Magento\Framework\HTTP\Header;

use Sunpay\SunpayPayment\Model\SunpayInvoice;

use Sunpay\SunpayPayment\Helper\Foundation\EncryptionsHelper;
use Sunpay\SunpayPayment\Helper\Services\Config\MainService;
use Sunpay\SunpayPayment\Helper\Services\Common\OrderService;

class InvoiceService extends AbstractHelper
{
    protected $_urlInterface;
    protected $_httpHelper;

    protected $_encryptionsHelper;
    protected $_mainService;
    protected $_orderService;

    public function __construct(
        Context $context,
        UrlInterface $urlInterface,
        Header $httpHelper,

        EncryptionsHelper $encryptionsHelper,
        MainService $mainService,
        OrderService $orderService,
    )
    {
        $this->_urlInterface = $urlInterface;
        $this->_httpHelper = $httpHelper;

        $this->_encryptionsHelper = $encryptionsHelper;
        $this->_mainService = $mainService;
        $this->_orderService = $orderService;

        parent::__construct($context);
    }

    /**
     * 取得發票開立類別名稱對應表
     *
     * @return array
     */
    public function getInvoiceTypeTable()
    {
        return [
            SunpayInvoice::SUNPAY_INVOICE_TYPE_B2C => '個人',
            SunpayInvoice::SUNPAY_INVOICE_TYPE_B2B => '公司',
        ];
    }

    /**
     * 開立發票
     * @param string $orderId
     */
    public function invoiceIssue($orderId)
    {
        //固定使用台灣時間(UTC+8)
		date_default_timezone_set("Asia/Taipei");
        // 判斷發票模組是否啟動
        $sunpayEnableInvoice = $this->_mainService->isInvoiceModuleEnable();
        $this->_logger->debug('InvoiceService sunpayEnableInvoice:'. print_r($sunpayEnableInvoice, true));
        if ($sunpayEnableInvoice == 0) {
            return [
                'code'  => '1003',
                'msg'   => __('code_1003'),
                'data'  => '',
            ];;
        }

        // 判斷是否已經開立過發票
        $sunpayInvoiceTag = $this->_orderService->getSunpayInvoiceTag($orderId);
        $this->_logger->debug('InvoiceService sunpayInvoiceTag:'. print_r($sunpayInvoiceTag, true));
        if ($sunpayInvoiceTag == 0) {
        } else {
            return [
                'code'  => '1002',
                'msg'   => __('code_1002'),
                'data'  => '',
            ];;
        }

        $status = false;

        // 取出是否為測試模式
        $invoiceStage = $this->_mainService->getPaymentConfig('enabled_payment_stage');
        $this->_logger->debug('InvoiceService invoiceStage:'. print_r($invoiceStage, true));

        // 取出 KEY IV MID
        $CompanyID  = $this->_mainService->getInvoiceConfig('ei_company_id'); // 商店統一編號
        $MerchantID = $this->_mainService->getInvoiceConfig('ei_merchant_id'); // 商店代號
        $key        = $this->_mainService->getInvoiceConfig('ei_hash_key');  // 商店專屬串接金鑰HashKey值
        $iv         = $this->_mainService->getInvoiceConfig('ei_hash_iv');  // 商店專屬串接iv

        $this->_logger->debug('InvoiceService CompanyID:'. print_r($CompanyID, true));
        $this->_logger->debug('InvoiceService MerchantID:'. print_r($MerchantID, true));
        $this->_logger->debug('InvoiceService key:'. print_r($key, true));
        $this->_logger->debug('InvoiceService iv:'. print_r($iv, true));

        //取出其餘發票設定
        $IsSendMessage = $this->_mainService->getInvoiceConfig('ei_is_send_message');
        $IsSendPaper    = $this->_mainService->getInvoiceConfig('ei_is_send_paper');

        $this->_logger->debug('InvoiceService IsSendMessage:'. print_r($IsSendMessage, true));
        $this->_logger->debug('InvoiceService IsSendPaper:'. print_r($IsSendPaper, true));

		// $CompanyID  = $this->companyID; // 商店統一編號
		// $MerchantID = $this->merchantID; // 商店代號
		// $key        = $this->HashKey;  // 商店專屬串接金鑰HashKey值
		// $iv         = $this->HashIV;  // 商店專屬串接iv

        //取得紅陽交易編號
        $tradeNo = $this->_orderService->getSunpayPaymentMerchantTradeNo($orderId);
        if(empty($tradeNo)){
            $tradeNo = $orderId;
        }
		// $discount_with_no_tax = $order->get_total_discount();
		$baseDiscountAmount    = $this->_orderService->getBaseDiscountAmount($orderId);
        $baseShippingAmount = $this->_orderService->getBaseShippingAmount($orderId);
        $baseSubtotal       = $this->_orderService->getBaseSubtotal($orderId);
        $baseGrandTotal     = $this->_orderService->getBaseGrandTotal($orderId);

        $itemPrice = $baseSubtotal + $baseShippingAmount + $baseDiscountAmount;
        $itemAmount = intval($baseGrandTotal);
        $saleAmount = intval(round($itemAmount, 0));

        $this->_logger->debug('InvoiceService baseDiscountAmount:'. print_r($baseDiscountAmount, true));
        $this->_logger->debug('InvoiceService baseShippingAmount:'. print_r($baseShippingAmount, true));
        $this->_logger->debug('InvoiceService baseSubtotal:'. print_r($baseSubtotal, true));
        $this->_logger->debug('InvoiceService saleAmount:'. print_r($saleAmount, true));

		// 商品資訊
        $productItems[] = array(
            "description" => '網路商品一批',
            "quantity" => 1,
            "unit" => "批",
            "unitPrice" => intval($baseSubtotal),
            "amount" => intval($baseSubtotal),
            "remark" => "",
            "taxType" => null,
        );	

		if ( $baseShippingAmount > 0 ) {
			$productItems[] = array(
				"description" => "運費",
				"quantity" => "1",
				"unit" => "個",
				"unitPrice" => intval($baseShippingAmount),
				"amount" => intval($baseShippingAmount),
				"remark" => "",
				"taxType" => null,
			);			
		}

		if ( $baseDiscountAmount > 0 ) {
			$productItems[] = array(
				"description" => "折扣",
				"quantity" => "1",
				"unit" => "次",
				"unitPrice" => intval($baseDiscountAmount),
				"amount" => intval($baseDiscountAmount),
				"remark" => "",
				"taxType" => null,
			);			
		}

        // 發票欄位資訊
        $sunpayInvoiceCarrierType        = $this->_orderService->getSunpayInvoiceCarruerType($orderId);  //carrierType
        $sunpayInvoiceType               = $this->_orderService->getsunpayInvoiceType($orderId);         //eiType
        $sunpayInvoiceCarruerNum         = $this->_orderService->getSunpayInvoiceCarruerNum($orderId);   //carrierId1
        $sunpayInvoiceLoveCode           = $this->_orderService->getSunpayInvoiceLoveCode($orderId);     //poban
        $sunpayInvoiceCustomerIdentifier = $this->_orderService->getSunpayInvoiceCustomerIdentifier($orderId); //buyerUBN
        $sunpayInvoiceCustomerCompany    = $this->_orderService->getSunpayInvoiceCustomerCompany($orderId);    //buyerName

        $this->_logger->debug('InvoiceService sunpayInvoiceCarrierType:'. print_r($sunpayInvoiceCarrierType, true));
        $this->_logger->debug('InvoiceService sunpayInvoiceType:'. print_r($sunpayInvoiceType, true));
        $this->_logger->debug('InvoiceService sunpayInvoiceCarruerNum:'. print_r($sunpayInvoiceCarruerNum, true));
        $this->_logger->debug('InvoiceService sunpayInvoiceLoveCode:'. print_r($sunpayInvoiceLoveCode, true));
        $this->_logger->debug('InvoiceService sunpayInvoiceCustomerIdentifier:'. print_r($sunpayInvoiceCustomerIdentifier, true));
        $this->_logger->debug('InvoiceService sunpayInvoiceCustomerCompany:'. print_r($sunpayInvoiceCustomerCompany, true));

        //紅陽交易編號
        // $sunpayTradeNo = 

		$totalAmt = $baseGrandTotal;

		$salesAmt = 0; //round($totalAmt / (1+$taxRate));
		$zeroAmt = 0;
		$freeAmt = 0;
		$customsClearanceMark = 0;
		$zeroTaxRateReason = null;
		$taxType           = $this->_mainService->getInvoiceConfig('ei_tax_type');

		switch ( $taxType ) {
			case 1:
				$invoiceType = 7;
				$taxRate = 0.05;
				$salesAmt = round($totalAmt / (1+$taxRate));
				break;
			case 2.1:
				$taxType          = 2;
				$taxRate = 0;
				$invoiceType = 7;
				$customsClearanceMark = 1;
				$zeroAmt = round($totalAmt / (1+$taxRate));
				$zeroTaxRateReason = 71;
				break;
			case 2.2:
				$taxType          = 2;
				$taxRate = 0;
				$invoiceType = 7;
				$customsClearanceMark = 2;
				$zeroAmt = round($totalAmt / (1+$taxRate));
				$zeroTaxRateReason = 71;
				break;
			case 3:
				$invoiceType = 7;
				$taxRate = 0;
				$freeAmt = round($totalAmt / (1+$taxRate));
				break;
			case 4:
				$taxRate = $this->_mainService->getInvoiceConfig('ei_tax_rate');
				$invoiceType = 8;
				$salesAmt = round($totalAmt / (1+$taxRate));
				break;
			case 9:
				$taxRate = null;
				$invoiceType = 7;
				$zeroTaxRateReason = 71;
				$salesAmt = round($totalAmt / (1+$taxRate));
				break;
		}

		// B2B可輸入買受人名稱 若無輸入就使用帳單的姓名(B2C直接用這個)
		$buyerName    = $this->_orderService->getBillingName($orderId);
		$buyerEmail   = $this->_orderService->getBillingEmail($orderId);
		$buyerPhone   = $this->_orderService->getBillingTelephone($orderId);
        // 取得帳單地址資訊
        // $billingCity        = $this->_orderService->getBillingCity($orderId);
        // $billingRegion      = $this->_orderService->getBillingRegion($orderId);
        $billingPostcode    = $this->_orderService->getBillingPostcode($orderId);
        $billingStreet      = $this->_orderService->getBillingStreet($orderId);
        // 地址郵遞區號組合
        $getPostalName = $this->getPostalName($billingPostcode);
		$buyerAddress = $getPostalName.$billingStreet;

        $this->_logger->debug('InvoiceService buyerName:'. print_r($buyerName, true));
        $this->_logger->debug('InvoiceService buyerPhone:'. print_r($buyerPhone, true));
        $this->_logger->debug('InvoiceService buyerEmail:'. print_r($buyerEmail, true));
        // $this->_logger->debug('InvoiceService billingCity:'. print_r($billingCity, true));
        // $this->_logger->debug('InvoiceService billingRegion:'. print_r($billingRegion, true));
        $this->_logger->debug('InvoiceService billingPostcode:'. print_r($billingPostcode, true));
        $this->_logger->debug('InvoiceService billingStreet:'. print_r($billingStreet, true));

		switch ( $sunpayInvoiceCarrierType ) {
			case 0:
				//無載具
				// $IsSendPaper = $IsSendPaper;
				// $sunpayInvoiceCarrierType = 0;
				$carrierId1  = null;
				$donateMark  = 0;
				$poban       = null;
				break;
			case 1:
				//手機條碼載具
				$IsSendPaper = 0;
				// $sunpayInvoiceCarrierType = 1;
				$carrierId1  = $sunpayInvoiceCarruerNum;
				$donateMark  = 0;
				$poban       = null;
				break;
			case 2:
				//自然人憑證條碼載具
				$IsSendPaper = 0;
				// $sunpayInvoiceCarrierType = 2;
				$carrierId1  = $sunpayInvoiceCarruerNum;
				$donateMark  = 0;
				$poban       = null;
				break;
			case 3:
				//紅陽會員載具
				$IsSendPaper = 0;
				// $sunpayInvoiceCarrierType = 3;
				$carrierId1  = null;
				$donateMark  = 0;
				$poban       = null;
				break;
			case 4:
				//捐贈發票
				$IsSendPaper = 0;
				$sunpayInvoiceCarrierType = 0;
				$carrierId1  = "";
				$donateMark  = 1;
				$poban       = $sunpayInvoiceLoveCode;
				break;
			default:
				//無載具
				$IsSendPaper = 0;
				$sunpayInvoiceCarrierType = 0;
				$carrierId1  = null;
				$donateMark  = 0;
				$poban       = null;
				break;
		}

		//API交易檢查碼
		$nowTimestamp = time();
		$tokenData = array(
			"CompanyID" => $CompanyID,
			// "TimeStamp" => strval($nowTimestamp),
			"TimeStamp" => strval(strtotime("+8 hours", $nowTimestamp)),
		);
		$tokenDataJson = json_encode($tokenData);
		$token = $this->_encryptionsHelper->aesEncrypt(urlencode($tokenDataJson), $key, $iv);

		if($sunpayInvoiceType == "B2B"){
			//B2B
			if ( $invoiceStage ) {
				$url = 'https://testinv.sunpay.com.tw/api/v1/SunPay/CreateInvoiceb2b'; // 測試網址
			} else {
				$url = 'https://einv.sunpay.com.tw/api/v1/SunPay/CreateInvoiceb2b'; // 正式網址
			}
			$post_data_array = array( // post_data欄位資料
				'merchantID'           => $MerchantID,
				'orderNo'              => $tradeNo,
				'buyerIdentifier'      => $sunpayInvoiceCustomerIdentifier,
				'buyerName'            => $buyerName,
				'buyerEmailAddress'    => $buyerEmail,
				'IsSendMessage'        => intval($IsSendMessage),
				'buyerTelephoneNumber' => $buyerPhone,
				'IsSendPaper'          => $IsSendPaper,
				'buyerAddress'         => $buyerAddress,
				'invoiceType'          => $invoiceType,
				'taxType'              => intval($taxType),
				'taxRate'        	   => $taxRate,
				'taxAmount'            => $totalAmt - $salesAmt,
				'salesAmount' 		   => $salesAmt,
				'zeroTaxSalesAmount'   => $zeroAmt,
				'freeTaxSalesAmount'   => $freeAmt,
				'totalAmount'          => $totalAmt,
				'mem'         		   => "",
				'customsClearanceMark' => $customsClearanceMark,
				'zeroTaxRateReason'    => $zeroTaxRateReason,
				'productItems'         => $productItems,
				'Token'          	   => $token,
			);
		}else{
			//B2C
			if ( $invoiceStage ) {
				$url = 'https://testinv.sunpay.com.tw/api/v1/SunPay/CreateInvoiceb2c'; // 測試網址
			} else {
				$url = 'https://einv.sunpay.com.tw/api/v1/SunPay/CreateInvoiceb2c'; // 正式網址
			}
			$post_data_array = array( // post_data欄位資料
				'merchantID'           => $MerchantID,
				'orderNo'              => $tradeNo,
				'buyerIdentifier'      => "",
				'buyerName'            => $buyerName,
				'buyerEmailAddress'    => $buyerEmail,
				'isSendMessage'        => intval($IsSendMessage),
				'buyerTelephoneNumber' => $buyerPhone,
				'isSendPaper'          => $IsSendPaper,
				'buyerAddress'         => $buyerAddress,
				'invoiceType'          => $invoiceType,
				'donateMark'     	   => $donateMark ?? 0,
				'poban'       		   => $poban,
				'carrierType'      	   => intval($sunpayInvoiceCarrierType),
				'carrierId1'           => $carrierId1,
				'taxType'              => intval($taxType),
				'taxRate'        	   => $taxRate,
				'taxAmount'            => $totalAmt - $salesAmt,
				'salesAmount' 		   => $salesAmt,
				'zeroTaxSalesAmount'   => $zeroAmt,
				'freeTaxSalesAmount'   => $freeAmt,
				'totalAmount'          => $totalAmt,
				'mem'         		   => "",
				'customsClearanceMark' => $customsClearanceMark,
				'zeroTaxRateReason'    => $zeroTaxRateReason,
				'isprint'         	   => $IsSendPaper,
				'productItems'         => $productItems,
				'token'          	   => $token,
			);
		}

		$post_data_json              = json_encode($post_data_array); 
        $this->_logger->debug('InvoiceService post_data_array:'. print_r($post_data_array, true));
        $this->_logger->debug('InvoiceService post_data_json:'. print_r($post_data_json, true));

		$result                 = $this->_curl( $url, $post_data_json, "POST" ); // 背景送出
        $this->_logger->debug('InvoiceService result:'. print_r($result, true));
		// Add order notes on admin
		$respondDecode = json_decode( $result['web_info'], true );
		if ( $respondDecode['status'] == "SUCCESS" ) {
            $resultDecode   = $respondDecode['result'];
            $this->_logger->debug('InvoiceService resultDecode:'. print_r($resultDecode, true));
            $tradeNumber = $resultDecode['tradeNumber'];
            $invoiceNumber  = $resultDecode['invoiceNumber'];

            // 更新訂單發票欄位
            $this->_orderService->setOrderData($orderId, 'sunpay_invoice_number', $invoiceNumber);
            if(isset($resultDecode['crT_DAT'])){
                $this->_orderService->setOrderData($orderId, 'sunpay_invoice_date', $resultDecode['crT_DAT']);
            }
            if(isset($resultDecode['randomNumber']) && is_numeric($resultDecode['randomNumber'])){
                $this->_orderService->setOrderData($orderId, 'sunpay_invoice_random_number', $resultDecode['randomNumber']);
            }

            $this->_orderService->setOrderData($orderId, 'sunpay_invoice_tag', 1);
            $this->_orderService->setOrderData($orderId, 'sunpay_invoice_issue_type', 1);
            $this->_orderService->setOrderData($orderId, 'sunpay_invoice_od_sob', $tradeNo);
            $this->_logger->debug('InvoiceService setOrderData Complete');

            $comment      = $respondDecode['message'] . '<br>紅陽開立序號: ' . $tradeNumber . '<br>' . '發票號碼: ' . $invoiceNumber;
            if(!empty($sunpayInvoiceCustomerIdentifier)){
                $comment .= "<br>統一編號: " .  $sunpayInvoiceCustomerIdentifier;
            }
            if(!empty($sunpayInvoiceCustomerCompany)){
                $comment .= "<br>買受人名稱: " .  $sunpayInvoiceCustomerCompany;
            }
            if($sunpayInvoiceType == "B2C" && !empty($sunpayInvoiceCarrierType)){
                switch($sunpayInvoiceCarrierType){
                    case 1:
                        $comment .= "<br>手機條碼載具: " .  $carrierId1;
                        break;
                    case 2:
                        $comment .= "<br>自然人憑證條碼載具: " .  $carrierId1;
                        break;
                    case 4:
                        $comment .= "<br>捐贈碼: " .  $poban;
                        break;
                    default:
                        break;
                }
            }
            $this->_logger->debug('InvoiceService comment:'. print_r($comment, true));
            $this->_orderService->setOrderCommentForBack($orderId, $comment);
            $this->_logger->debug('InvoiceService setOrderCommentForBack Complete');

            // 執行Magento自己的發票程序
            $this->_orderService->setOrderInvoice($orderId); 
            $this->_logger->debug('InvoiceService setOrderInvoice Complete');           

            // 組合回傳前端的dataResponse
            $dataResponse = [
                'sunpay_invoice_number'              => $invoiceNumber,
                'sunpay_invoice_tag'                 => 1,
                'sunpay_invoice_issue_type'          => 1,
                'sunpay_invoice_od_sob'              => $tradeNo,
                'sunpay_invoice_type'                => $this->getInvoiceTypeTable()[$sunpayInvoiceType],
            ];
            if(isset($resultDecode['crT_DAT'])){
                $dataResponse['sunpay_invoice_date'] = $resultDecode['crT_DAT'];
            }else{
                $dataResponse['sunpay_invoice_date'] = "";
            }
            if(isset($resultDecode['randomNumber'])){
                $dataResponse['sunpay_invoice_random_number'] = $resultDecode['randomNumber'];
            }else{
                $dataResponse['sunpay_invoice_random_number'] = "";
            }
            if(isset($this->getInvoiceTypeTable()[$sunpayInvoiceType])){
                $dataResponse['sunpay_invoice_type'] = $this->getInvoiceTypeTable()[$sunpayInvoiceType];
            }else{
                $dataResponse['sunpay_invoice_type'] = $sunpayInvoiceType;
            }

            switch ($sunpayInvoiceType) {
                case SunpayInvoice::SUNPAY_INVOICE_TYPE_B2B:
                    $dataResponse['sunpay_invoice_customer_company'] = $sunpayInvoiceCustomerCompany ;
                    $dataResponse['sunpay_invoice_customer_identifier'] = $sunpayInvoiceCustomerIdentifier ;
                    break;
                case SunpayInvoice::SUNPAY_INVOICE_TYPE_B2C:
                    if($sunpayInvoiceCarrierType == 1 || $sunpayInvoiceCarrierType == 2){
                        $dataResponse['sunpay_invoice_carruer_num'] = $sunpayInvoiceCarruerNum ;
                    }elseif($sunpayInvoiceCarrierType == 4){
                        $dataResponse['sunpay_invoice_love_code'] = $sunpayInvoiceLoveCode ;
                    }
                    break;
            }
            $this->_logger->debug('InvoiceService dataResponse:'. print_r($dataResponse, true));

            return [
                'code' => '0999',
                'msg'   => __('code_0999'),
                'data' => $dataResponse
            ];

		} else {
            $comment = '發票開立失敗<br>錯誤訊息：' . $respondDecode['errors'][0];
            $this->_logger->debug('InvoiceService comment:'. print_r($comment, true));
            $this->_orderService->setOrderCommentForBack($orderId, $comment);
		}
    }

    /**
     * 取出區域
     * @param  int  $zipCode
     * @return string  $return
     */
    public function getPostalName($zipCode = 0)
    {
        $address = [
            0   => '',
            100 => '臺北市中正區',
            103 => '臺北市大同區',
            104 => '臺北市中山區',
            105 => '臺北市松山區',
            106 => '臺北市大安區',
            108 => '臺北市萬華區',
            110 => '臺北市信義區',
            111 => '臺北市士林區',
            112 => '臺北市北投區',
            114 => '臺北市內湖區',
            115 => '臺北市南港區',
            116 => '臺北市文山區',
            200 => '基隆市仁愛區',
            201 => '基隆市信義區',
            202 => '基隆市中正區',
            203 => '基隆市中山區',
            204 => '基隆市安樂區',
            205 => '基隆市暖暖區',
            206 => '基隆市七堵區',
            207 => '新北市萬里區',
            208 => '新北市金山區',
            220 => '新北市板橋區',
            221 => '新北市汐止區',
            222 => '新北市深坑區',
            223 => '新北市石碇區',
            224 => '新北市瑞芳區',
            226 => '新北市平溪區',
            227 => '新北市雙溪區',
            228 => '新北市貢寮區',
            231 => '新北市新店區',
            232 => '新北市坪林區',
            233 => '新北市烏來區',
            234 => '新北市永和區',
            235 => '新北市中和區',
            236 => '新北市土城區',
            237 => '新北市三峽區',
            238 => '新北市樹林區',
            239 => '新北市鶯歌區',
            241 => '新北市三重區',
            242 => '新北市新莊區',
            243 => '新北市泰山區',
            244 => '新北市林口區',
            247 => '新北市蘆洲區',
            248 => '新北市五股區',
            249 => '新北市八里區',
            251 => '新北市淡水區',
            252 => '新北市三芝區',
            253 => '新北市石門區',
            260 => '宜蘭縣宜蘭市',
            261 => '宜蘭縣頭城鎮',
            262 => '宜蘭縣礁溪鄉',
            263 => '宜蘭縣壯圍鄉',
            264 => '宜蘭縣員山鄉',
            265 => '宜蘭縣羅東鎮',
            266 => '宜蘭縣三星鄉',
            267 => '宜蘭縣大同鄉',
            268 => '宜蘭縣五結鄉',
            269 => '宜蘭縣冬山鄉',
            270 => '宜蘭縣蘇澳鎮',
            272 => '宜蘭縣南澳鄉',
            300 => '新竹市',
            302 => '新竹縣竹北市',
            303 => '新竹縣湖口鄉',
            304 => '新竹縣新豐鄉',
            305 => '新竹縣新埔鎮',
            306 => '新竹縣關西鎮',
            307 => '新竹縣芎林鄉',
            308 => '新竹縣寶山鄉',
            310 => '新竹縣竹東鎮',
            311 => '新竹縣五峰鄉',
            312 => '新竹縣橫山鄉',
            313 => '新竹縣尖石鄉',
            314 => '新竹縣北埔鄉',
            315 => '新竹縣峨眉鄉',
            320 => '桃園縣中壢市',
            324 => '桃園縣平鎮市',
            325 => '桃園縣龍潭鄉',
            326 => '桃園縣楊梅鎮',
            327 => '桃園縣新屋鄉',
            328 => '桃園縣觀音鄉',
            330 => '桃園縣桃園市',
            333 => '桃園縣龜山鄉',
            334 => '桃園縣八德市',
            335 => '桃園縣大溪鎮',
            336 => '桃園縣復興鄉',
            337 => '桃園縣大園鄉',
            338 => '桃園縣蘆竹鄉',
            350 => '苗栗縣竹南鎮',
            351 => '苗栗縣頭份鎮',
            352 => '苗栗縣三灣鄉',
            353 => '苗栗縣南庄鄉',
            354 => '苗栗縣獅潭鄉',
            356 => '苗栗縣後龍鎮',
            357 => '苗栗縣通霄鎮',
            358 => '苗栗縣苑裡鎮',
            360 => '苗栗縣苗栗市',
            361 => '苗栗縣造橋鄉',
            362 => '苗栗縣頭屋鄉',
            363 => '苗栗縣公館鄉',
            364 => '苗栗縣大湖鄉',
            365 => '苗栗縣泰安鄉',
            366 => '苗栗縣銅鑼鄉',
            367 => '苗栗縣三義鄉',
            368 => '苗栗縣西湖鄉',
            369 => '苗栗縣卓蘭鎮',
            400 => '臺中市中區',
            401 => '臺中市東區',
            402 => '臺中市南區',
            403 => '臺中市西區',
            404 => '臺中市北區',
            406 => '臺中市北屯區',
            407 => '臺中市西屯區',
            408 => '臺中市南屯區',
            411 => '臺中市太平區',
            412 => '臺中市大里區',
            413 => '臺中市霧峰區',
            414 => '臺中市烏日區',
            420 => '臺中市豐原區',
            421 => '臺中市后里區',
            422 => '臺中市石岡區',
            423 => '臺中市東勢區',
            424 => '臺中市和平區',
            426 => '臺中市新社區',
            427 => '臺中市潭子區',
            428 => '臺中市大雅區',
            429 => '臺中市神岡區',
            432 => '臺中市大肚區',
            433 => '臺中市沙鹿區',
            434 => '臺中市龍井區',
            435 => '臺中市梧棲區',
            436 => '臺中市清水區',
            437 => '臺中市大甲區',
            438 => '臺中市外埔區',
            439 => '臺中市大安區',
            500 => '彰化縣彰化市',
            502 => '彰化縣芬園鄉',
            503 => '彰化縣花壇鄉',
            504 => '彰化縣秀水鄉',
            505 => '彰化縣鹿港鎮',
            506 => '彰化縣福興鄉',
            507 => '彰化縣線西鄉',
            508 => '彰化縣和美鎮',
            509 => '彰化縣伸港鄉',
            510 => '彰化縣員林鎮',
            511 => '彰化縣社頭鄉',
            512 => '彰化縣永靖鄉',
            513 => '彰化縣埔心鄉',
            514 => '彰化縣溪湖鎮',
            515 => '彰化縣大村鄉',
            516 => '彰化縣埔鹽鄉',
            520 => '彰化縣田中鎮',
            521 => '彰化縣北斗鎮',
            522 => '彰化縣田尾鄉',
            523 => '彰化縣埤頭鄉',
            524 => '彰化縣溪州鄉',
            525 => '彰化縣竹塘鄉',
            526 => '彰化縣二林鎮',
            527 => '彰化縣大城鄉',
            528 => '彰化縣芳苑鄉',
            530 => '彰化縣二水鄉',
            540 => '南投縣南投市',
            541 => '南投縣中寮鄉',
            542 => '南投縣草屯鎮',
            544 => '南投縣國姓鄉',
            545 => '南投縣埔里鎮',
            546 => '南投縣仁愛鄉',
            551 => '南投縣名間鄉',
            552 => '南投縣集集鎮',
            553 => '南投縣水里鄉',
            555 => '南投縣魚池鄉',
            556 => '南投縣信義鄉',
            557 => '南投縣竹山鎮',
            558 => '南投縣鹿谷鄉',
            600 => '嘉義市',
            602 => '嘉義縣番路鄉',
            603 => '嘉義縣梅山鄉',
            604 => '嘉義縣竹崎鄉',
            605 => '嘉義縣阿里山',
            606 => '嘉義縣中埔鄉',
            607 => '嘉義縣大埔鄉',
            608 => '嘉義縣水上鄉',
            611 => '嘉義縣鹿草鄉',
            612 => '嘉義縣太保市',
            613 => '嘉義縣朴子市',
            614 => '嘉義縣東石鄉',
            615 => '嘉義縣六腳鄉',
            616 => '嘉義縣新港鄉',
            621 => '嘉義縣民雄鄉',
            622 => '嘉義縣大林鎮',
            623 => '嘉義縣溪口鄉',
            624 => '嘉義縣義竹鄉',
            625 => '嘉義縣布袋鎮',
            630 => '雲林縣斗南鎮',
            631 => '雲林縣大埤鄉',
            632 => '雲林縣虎尾鎮',
            633 => '雲林縣土庫鎮',
            634 => '雲林縣褒忠鄉',
            635 => '雲林縣東勢鄉',
            636 => '雲林縣臺西鄉',
            637 => '雲林縣崙背鄉',
            638 => '雲林縣麥寮鄉',
            640 => '雲林縣斗六市',
            643 => '雲林縣林內鄉',
            646 => '雲林縣古坑鄉',
            647 => '雲林縣莿桐鄉',
            648 => '雲林縣西螺鎮',
            649 => '雲林縣二崙鄉',
            651 => '雲林縣北港鎮',
            652 => '雲林縣水林鄉',
            653 => '雲林縣口湖鄉',
            654 => '雲林縣四湖鄉',
            655 => '雲林縣元長鄉',
            700 => '臺南市中西區',
            701 => '臺南市東區',
            702 => '臺南市南區',
            704 => '臺南市北區',
            708 => '臺南市安平區',
            709 => '臺南市安南區',
            710 => '臺南市永康區',
            711 => '臺南市歸仁區',
            712 => '臺南市新化區',
            713 => '臺南市左鎮區',
            714 => '臺南市玉井區',
            715 => '臺南市楠西區',
            716 => '臺南市南化區',
            717 => '臺南市仁德區',
            718 => '臺南市關廟區',
            719 => '臺南市龍崎區',
            720 => '臺南市官田區',
            721 => '臺南市麻豆區',
            722 => '臺南市佳里區',
            723 => '臺南市西港區',
            724 => '臺南市七股區',
            725 => '臺南市將軍區',
            726 => '臺南市學甲區',
            727 => '臺南市北門區',
            730 => '臺南市新營區',
            731 => '臺南市後壁區',
            732 => '臺南市白河區',
            733 => '臺南市東山區',
            734 => '臺南市六甲區',
            735 => '臺南市下營區',
            736 => '臺南市柳營區',
            737 => '臺南市鹽水區',
            741 => '臺南市善化區',
            742 => '臺南市大內區',
            743 => '臺南市山上區',
            744 => '臺南市新市區',
            745 => '臺南市安定區',
            800 => '高雄市新興區',
            801 => '高雄市前金區',
            802 => '高雄市苓雅區',
            803 => '高雄市鹽埕區',
            804 => '高雄市鼓山區',
            805 => '高雄市旗津區',
            806 => '高雄市前鎮區',
            807 => '高雄市三民區',
            811 => '高雄市楠梓區',
            812 => '高雄市小港區',
            813 => '高雄市左營區',
            814 => '高雄市仁武區',
            815 => '高雄市大社區',
            820 => '高雄市岡山區',
            821 => '高雄市路竹區',
            822 => '高雄市阿蓮區',
            823 => '高雄市田寮區',
            824 => '高雄市燕巢區',
            825 => '高雄市橋頭區',
            826 => '高雄市梓官區',
            827 => '高雄市彌陀區',
            828 => '高雄市永安區',
            829 => '高雄市湖內區',
            830 => '高雄市鳳山區',
            831 => '高雄市大寮區',
            832 => '高雄市林園區',
            833 => '高雄市鳥松區',
            840 => '高雄市大樹區',
            842 => '高雄市旗山區',
            843 => '高雄市美濃區',
            844 => '高雄市六龜區',
            845 => '高雄市內門區',
            846 => '高雄市杉林區',
            847 => '高雄市甲仙區',
            848 => '高雄市桃源區',
            849 => '高雄市那瑪夏區',
            851 => '高雄市茂林區',
            852 => '高雄市茄萣區',
            880 => '澎湖縣馬公市',
            881 => '澎湖縣西嶼鄉',
            882 => '澎湖縣望安鄉',
            883 => '澎湖縣七美鄉',
            884 => '澎湖縣白沙鄉',
            885 => '澎湖縣湖西鄉',
            900 => '屏東縣屏東市',
            901 => '屏東縣三地門',
            902 => '屏東縣霧臺鄉',
            903 => '屏東縣瑪家鄉',
            904 => '屏東縣九如鄉',
            905 => '屏東縣里港鄉',
            906 => '屏東縣高樹鄉',
            907 => '屏東縣鹽埔鄉',
            908 => '屏東縣長治鄉',
            909 => '屏東縣麟洛鄉',
            911 => '屏東縣竹田鄉',
            912 => '屏東縣內埔鄉',
            913 => '屏東縣萬丹鄉',
            920 => '屏東縣潮州鎮',
            921 => '屏東縣泰武鄉',
            922 => '屏東縣來義鄉',
            923 => '屏東縣萬巒鄉',
            924 => '屏東縣崁頂鄉',
            925 => '屏東縣新埤鄉',
            926 => '屏東縣南州鄉',
            927 => '屏東縣林邊鄉',
            928 => '屏東縣東港鎮',
            929 => '屏東縣琉球鄉',
            931 => '屏東縣佳冬鄉',
            932 => '屏東縣新園鄉',
            940 => '屏東縣枋寮鄉',
            941 => '屏東縣枋山鄉',
            942 => '屏東縣春日鄉',
            943 => '屏東縣獅子鄉',
            944 => '屏東縣車城鄉',
            945 => '屏東縣牡丹鄉',
            946 => '屏東縣恆春鎮',
            947 => '屏東縣滿州鄉',
            950 => '臺東縣臺東市',
            951 => '臺東縣綠島鄉',
            952 => '臺東縣蘭嶼鄉',
            953 => '臺東縣延平鄉',
            954 => '臺東縣卑南鄉',
            955 => '臺東縣鹿野鄉',
            956 => '臺東縣關山鎮',
            957 => '臺東縣海端鄉',
            958 => '臺東縣池上鄉',
            959 => '臺東縣東河鄉',
            961 => '臺東縣成功鎮',
            962 => '臺東縣長濱鄉',
            963 => '臺東縣太麻里',
            964 => '臺東縣金峰鄉',
            965 => '臺東縣大武鄉',
            966 => '臺東縣達仁鄉',
            970 => '花蓮縣花蓮市',
            971 => '花蓮縣新城鄉',
            972 => '花蓮縣秀林鄉',
            973 => '花蓮縣吉安鄉',
            974 => '花蓮縣壽豐鄉',
            975 => '花蓮縣鳳林鎮',
            976 => '花蓮縣光復鄉',
            977 => '花蓮縣豐濱鄉',
            978 => '花蓮縣瑞穗鄉',
            979 => '花蓮縣萬榮鄉',
            981 => '花蓮縣玉里鎮',
            982 => '花蓮縣卓溪鄉',
            983 => '花蓮縣富里鄉',
            890 => '金門縣金沙鎮',
            891 => '金門縣金湖鎮',
            892 => '金門縣金寧鄉',
            893 => '金門縣金城鎮',
            894 => '金門縣烈嶼鄉',
            896 => '金門縣烏坵鄉',
            209 => '連江縣南竿鄉',
            210 => '連江縣北竿鄉',
            211 => '連江縣莒光鄉',
            212 => '連江縣東引鄉',
            817 => '南海諸島東沙',
            819 => '南海諸島南沙',
            290 => '釣魚台列嶼'
        ];

        return (isset($address[$zipCode])) ? $address[$zipCode] : '' ;
    }

    function _curl($url = '', $parameter = '', $post_type = '')
    {
        $curl_options = array(
            CURLOPT_URL => $url,
            CURLOPT_HEADER => false,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_USERAGENT => $this->_httpHelper->getHttpUserAgent(),
            CURLOPT_SSL_VERIFYPEER => TRUE,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,
        );
        $this->_logger->debug('InvoiceService curl_options:'. print_r($curl_options, true));

        if ($post_type == 'POST') {
            $curl_options[CURLOPT_POST] = "1";
            $curl_options[CURLOPT_HTTPHEADER] = [
                'Content-Type: application/json',
                'Accept: application/json'
            ];
            $curl_options[CURLOPT_POSTFIELDS] = $parameter;
        }

        $ch = curl_init();
        curl_setopt_array($ch, $curl_options);
        $result = curl_exec($ch);
        $retcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_errno($ch);

        curl_close($ch);

        $return_info = array(
            "url" => $url,
            "sent_parameter" => $parameter,
            "http_status" => $retcode,
            "curl_error_no" => $curl_error,
            "web_info" => $result,
        );

        return $return_info;
    }
}