<?php
namespace Sunpay\SunpayPayment\Controller\Process;

use Exception;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Sales\Model\Order;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Psr\Log\LoggerInterface ;
use Sunpay\SunpayPayment\Controller\Api\Invoice;
use Sunpay\SunpayPayment\Helper\Services\Common\EncryptionsService;
use Sunpay\SunpayPayment\Helper\Services\Common\OrderService;
use Sunpay\SunpayPayment\Helper\Services\Config\MainService;
use Sunpay\SunpayPayment\Helper\Services\Config\PaymentService;
use Sunpay\SunpayPayment\Helper\Services\Config\InvoiceService;
use Sunpay\SunpayPayment\Helper\Foundation\EncryptionsHelper;
use Sunpay\SunpayPayment\Helper\Foundation\SunpayPaymentHelper;

class PaymentResponse extends Action implements CsrfAwareActionInterface
{

    protected $_loggerInterface;
    protected $_requestInterface;

    protected $_encryptionsService;
    protected $_orderService;

    protected $_mainService;
    protected $_paymentService;
    protected $_invoiceService;

    protected $_encryptionsHelper;
    protected $_sunpaypaymentHelper;

    public function __construct(
        LoggerInterface $loggerInterface,
        Context $context,
        RequestInterface $requestInterface,

        EncryptionsService $encryptionsService,
        OrderService $orderService,

        MainService $mainService,
        PaymentService $paymentService,
        InvoiceService $invoiceService,

        EncryptionsHelper $encryptionsHelper,
        SunpayPaymentHelper $sunpaypaymentHelper
    )
    {
        $this->_loggerInterface = $loggerInterface;
        $this->_requestInterface = $requestInterface;

        $this->_encryptionsService = $encryptionsService;
        $this->_orderService = $orderService;

        $this->_mainService = $mainService;
        $this->_paymentService = $paymentService;
        $this->_invoiceService = $invoiceService;

        $this->_encryptionsHelper = $encryptionsHelper;
        $this->_sunpaypaymentHelper = $sunpaypaymentHelper;

        return parent::__construct($context);
    }

    public function execute()
    {
        // 接收金流資訊
        $paymentInfo = $this->_requestInterface->getPostValue();
        $this->_loggerInterface->debug('PaymentResponse paymentInfo:'. print_r($paymentInfo,true));

        if (!isset($paymentInfo['rsamsg'])) {
            $this->_loggerInterface->debug('PaymentResponse paymentInfo: Get SunPay feedback failed.');
            throw new Exception('Get SunPay feedback failed.');
        } else {

            $req_data = array();
            $paymentMerchantId = $this->_mainService->getPaymentConfig('payment_mid');
            $paymentRSAPublicKey    = $this->_mainService->getPaymentConfig('payment_rsakey');
            $paymentSHA2Key     = $this->_mainService->getPaymentConfig('payment_sha2key');
            if (!empty($this->_paymentService->sanitize_text_field($paymentInfo['rsamsg']))) {
                $req_data = $this->_encryptionsHelper->get_decrypted_data(
                    $this->_paymentService->sanitize_text_field($paymentInfo['rsamsg']),
                    $paymentRSAPublicKey,
                    $paymentSHA2Key
                );
                if (!$this->_encryptionsHelper->chkShaIsVaildByReturnData($paymentInfo, $req_data['check_value'], $paymentMerchantId)) {
                    $this->_loggerInterface->debug('PaymentResponse Error Message: 檢查碼錯誤');
                    throw new Exception("檢查碼錯誤");
                }
                if (!is_array($req_data) || 
                (isset($req_data['data']) && !is_array($req_data['data']))) {
                    $this->_loggerInterface->debug('PaymentResponse Error Message: 解密失敗');
                    throw new Exception("解密失敗");
                }
            }
            $this->_loggerInterface->debug('PaymentResponse req_data:'. print_r($req_data,true));

            // 初始化$req_data 避免因index不存在導致NOTICE 若無傳入值的index將設為null
            $init_indexes_head  = 'send_time,web';
            $head = $this->_encryptionsHelper->init_array_data($req_data['data']['head'],$init_indexes_head);
            $init_indexes_body  = 'approve_code,card_no,card_type,currency,mn,name,note1,note2,pay_agency,pay_agency_memo,
                                pay_date,pay_result,pay_time,td,trade_no,save_card_token_result,user_no';                                                     //付款完成之回傳參數欄位
            $init_indexes_body .= 'delivery_type,store_id,store_name,store_type,lgs_type,receiver_name,receiver_phone';                                          //超商物流之回傳參數欄位
            $init_indexes_body .= 'account_id,account_name,bank_code,banc_name,due_date,pay_code,barcodeA,barcodeB,barcodeC';                                    //取號完成之回傳參數欄位
            $body = $this->_encryptionsHelper->init_array_data($req_data['data']['body'],$init_indexes_body);
            $req_data = array(
                "body" => $body,
                "head" => $head
            );

            $orderId = trim($req_data['body']['td']);
            $re_Status  = trim($req_data['body']['pay_result']) != '' ? trim($req_data['body']['pay_result']) : null;
            $re_TradeNo = $req_data['body']['trade_no'];
            $re_Amt     = $req_data['body']['mn'];
            $re_CardType = $req_data['body']['card_type'];

            // 檢查回傳狀態是否為成功
            $msg = "";
            if ($re_Status != "10") {
                $msg = '訂單處理失敗: ';
                if($re_Status == "12" && $re_CardType == "09"){
                    //超商取貨另外處理
                    $msg = "付款方式:" . $this->_paymentService->getSunpayPaymentMethodNameByCardType($re_CardType);
                    $msg .= "，訂單已建立,等待付款";
                    $this->_orderService->setOrderCommentForBack($orderId, $msg);
                }else{
                    if($re_Status == "12" && str_contains("06,07,08",$re_CardType)){
                        //非即時性付款
                        $msg = "付款方式:" . $this->_paymentService->getSunpayPaymentMethodNameByCardType($re_CardType);
                        $msg .= "，訂單已建立,等待付款";
                        $this->_orderService->setOrderCommentForBack($orderId, $msg);
                    }else if($re_Status == "11"){
                        $msg .= "交易失敗";
                        $this->_orderService->setOrderCommentForBack($orderId, '錯誤訊息: ' . $msg);
                        $this->_orderService->setOrderState($orderId, Order::STATE_CANCELED);
                        $this->_orderService->setOrderStatus($orderId, Order::STATE_CANCELED);
                    }else if($re_Status == "12"){
                        $msg .= "訂單已建立過";
                        $this->_orderService->setOrderCommentForBack($orderId, '錯誤訊息: ' . $msg);
                        $this->_orderService->setOrderState($orderId, Order::STATE_CANCELED);
                        $this->_orderService->setOrderStatus($orderId, Order::STATE_CANCELED);
                    }
                    $this->_loggerInterface->debug('PaymentResponse Debug Message:'. $this->_paymentService->esc_attr($msg));
                    throw new Exception($this->_paymentService->esc_attr($msg));
                }
            }

            $orderCurrencyCode = $this->_orderService->getOrderCurrencyCode($orderId);
            $orderTotal = (!$orderCurrencyCode) ? $this->_orderService->getGrandTotal($orderId) : $this->_orderService->getBaseGrandTotal($orderId);
            if (!$this->_paymentService->validAmount($re_Amt, $orderTotal)) {
                $this->_loggerInterface->debug('PaymentResponse $paymentInfo.TradeAmt:'. print_r($paymentInfo['TradeAmt'], true));
                $this->_loggerInterface->debug('PaymentResponse $orderTotal:'. print_r($orderTotal, true));
                $this->_orderService->setOrderCommentForBack($orderId, '錯誤訊息: ' . '付款訂單金額不一致');
                $this->_orderService->setOrderState($orderId, Order::STATE_CANCELED);
                $this->_orderService->setOrderStatus($orderId, Order::STATE_CANCELED);
                $this->_loggerInterface->debug('PaymentResponse Error Message: Order amount are not identical.');
                throw new Exception('Order amount are not identical.');
            } else {

                // 訂單狀態判斷
                $createStatus = [Order::STATE_PENDING_PAYMENT, 'sunpay_pending_payment'];
                $orderStatus = $this->_orderService->getStatus($orderId);
                $this->_loggerInterface->debug('PaymentResponse $orderStatus:'. print_r($orderStatus, true));

                // 付款完成標籤 0.未付款完成 1.付款完成
                $paymentCompleteFlag = $this->_orderService->getSunpayPaymentCompleteTag($orderId);

                // 非即時性付款(超商取貨付款另外處理)
                if(str_contains("06,07,08",$re_CardType)){
                    try{
                        // 檢查是否付款
                        if (empty($req_data['body']['pay_date']) || empty($req_data['body']['pay_time'])) {
                            $comment = '訂單付款方式屬非即時性付款,尚未付款';
                            $this->_orderService->setOrderCommentForBack($orderId, $comment);
                            throw new Exception($this->_paymentService->esc_attr($comment));
                        }
                    }catch(Exception $e){
                        $msg .= '處理'.$this->_paymentService->getSunpayPaymentMethodNameByCardType($re_CardType).'失敗,';
                        $msg .= '錯誤訊息:' . $e->getMessage();
                        $this->_orderService->setOrderCommentForBack($orderId, $msg);
                        $this->_orderService->setOrderState($orderId, Order::STATE_CANCELED);
                        $this->_orderService->setOrderStatus($orderId, Order::STATE_CANCELED);
                        throw new Exception($this->_paymentService->esc_attr($msg));
                    }            
                }

                // 超商取貨
                if (!empty($req_data['body']['receiver_name']) || !empty($req_data['body']['store_name']) || !empty($req_data['body']['store_id']) || !empty($req_data['body']['store_type']) && !$paymentCompleteFlag) {
                    $storeType = urldecode($req_data['body']['store_type']); // 門市類型: 1:7-11, 2:全家, 3:OK, 4:萊爾富
                    $storeName = urldecode($req_data['body']['store_name']); // 門市名稱
                    $storeId = urldecode($req_data['body']['store_id']); // 門市編號
                    $name      = urldecode($req_data['body']['receiver_name']); // 取貨人姓名
                    $phone     = $req_data['body']['receiver_phone'];
                    $storeTypeStr = array(
                        '1' => '7-11',
                        '2' => '全家',
                        '3' => 'OK',
                        '4' => '萊爾富',
                    );
                    // 訂單備註
                    $comment = '取貨超商：' . $storeTypeStr[$storeType];
                    $comment .= '，取貨門市：' . $storeName;
                    $comment .= '，取貨門市ID：' . $storeId;
                    $comment .= '，取貨人姓名：' . $name;
                    $comment .= '，取貨人電話：' . $phone;

                    $this->_orderService->setOrderCommentForBack($orderId, $comment);
                }

                // 訂單處理
                if (in_array($orderStatus, $createStatus) && !$paymentCompleteFlag) {

                    if (!empty($req_data['body']['pay_date']) && !empty($req_data['body']['pay_time'])) {
                        //寫入付款時間
                        $paid_date = date("Y/m/d",strtotime($req_data['body']['pay_date'])) . " " . $req_data['body']['pay_time'];
                        $comment = '付款成功';
                        $comment .= '，付款時間:' . $paid_date;
                        //判斷是否為分期付款
                        $comment_card_type = $re_CardType;
                        $comment_ins = "";
                        if($re_CardType == '01' && !empty($req_data['body']['first_amt']) && !empty($req_data['body']['install_amt'])){
                            $comment_card_type = '01-2';
                            $comment_ins = "，頭期金額:" . $req_data['body']['first_amt'] . "，每期金額:" . $req_data['body']['install_amt'];
                        }
                        $comment .= '，付款方式:' . $this->_paymentService->getSunpayPaymentMethodNameByCardType($comment_card_type);
                        if($comment_ins != ""){
                            $comment .= $comment_ins;
                        }

                        // 全部確認過後，修改訂單狀態(處理中，並寄通知信)
                        $this->_orderService->setOrderCommentForBack($orderId, $comment);
                        $this->_orderService->setOrderState($orderId, Order::STATE_PROCESSING);
                        $this->_orderService->setOrderStatus($orderId, Order::STATE_PROCESSING);
                        $msg   = '訂單修改成功';

                        // 異動旗標
                        $this->_orderService->setOrderData($orderId, 'sunpay_payment_complete_tag', 1) ;
                        $this->_orderService->setOrderData($orderId, 'sunpay_payment_merchant_trade_no', $re_TradeNo);

                        //開立發票
                        try{
                            $this->_invoiceService->invoiceIssue($orderId);
                        }catch(Exception $e){
                            $this->_loggerInterface->debug('PaymentResponse Error Message: 開立發票失敗');
                        }

                    }

                }else{
                    $this->_loggerInterface->debug('PaymentResponse Error Message: 訂單已付款');
                    throw new Exception('訂單已付款');
                }
            }
        }

        return 'success' ;
    }

    /**
     * @inheritDoc
     */
    public function createCsrfValidationException(
        RequestInterface $requestInterface
    ): ?InvalidRequestException {

        return null;
    }

    /**
     * @inheritDoc
     */
    public function validateForCsrf(RequestInterface $requestInterface): ?bool
    {
        return true;
    }
}