<?php 
namespace Sunpay\SunpayPayment\Controller\Api;

use Psr\Log\LoggerInterface ;
use Magento\Framework\UrlInterface;
use Magento\Framework\Webapi\Exception;

use Sunpay\SunpayPayment\Model\SunpayInvoice;
use Sunpay\SunpayPayment\Helper\Services\Common\EncryptionsService;
use Sunpay\SunpayPayment\Helper\Services\Common\OrderService;
use Sunpay\SunpayPayment\Helper\Services\Config\MainService;
use Sunpay\SunpayPayment\Helper\Services\Config\InvoiceService;
use Sunpay\SunpayPayment\Helper\Services\Config\PaymentService;
use Sunpay\SunpayPayment\Helper\Foundation\SunpayPaymentHelper;

use Sunpay\Sdk\Factories\Factory;

class Invoice {

    protected $_loggerInterface;
    protected $_urlInterface;

    protected $_encryptionsService;
    protected $_orderService;
    protected $_mainService;
    protected $_invoiceService;
    protected $_paymentService;

    protected $_encryptionsHelper;
    protected $_generalHelper;

    public function __construct(
        LoggerInterface $loggerInterface,
        UrlInterface $urlInterface,

        EncryptionsService $encryptionsService,
        OrderService $orderService,
        MainService $mainService,
        InvoiceService $invoiceService,
        PaymentService $paymentService,
        SunpayPaymentHelper $generalHelper
    )
    {
        $this->_loggerInterface = $loggerInterface;
        $this->_urlInterface = $urlInterface;

        $this->_encryptionsService = $encryptionsService;
        $this->_orderService = $orderService;
        $this->_mainService = $mainService;
        $this->_invoiceService = $invoiceService;
        $this->_paymentService = $paymentService;
        
        $this->_generalHelper = $generalHelper;
    }

    /**
     * {@inheritdoc}
     */
    public function checkBarcode($barcode)
    {
        // 手機條碼格式驗證
        if (!preg_match('/^\/{1}[0-9a-zA-Z+-.]{7}$/', $barcode)) {
            throw new Exception(__('code_1008'), '1008');
        }

        // 取出是否為測試模式
        $invoiceStage = $this->_mainService->getInvoiceConfig('enabled_invoice_stage');
        $this->_loggerInterface->debug('getInvoiceAccountInfo invoiceStage:' . print_r($invoiceStage, true));

        // 取得發票會員資訊
        $accountInfo = $this->getInvoiceAccountInfo($invoiceStage);

        $factory = new Factory([
            'hashKey'   => $accountInfo['HashKey'],
            'hashIv'    => $accountInfo['HashIv'],
        ]);

        $postService = $factory->create('PostWithAesJsonResponseService');

        // 取出 URL
        $apiUrl = $this->_invoiceService->getApiUrl('check_barcode', $invoiceStage);
        $this->_loggerInterface->debug('checkBarcode apiUrl:'. print_r($apiUrl, true));

        // 組合送綠界格式
        $data = [
            'MerchantID' 	=> $accountInfo['MerchantId'],
            'BarCode' 		=> $barcode,
        ];

        $input = [
            'MerchantID' => $accountInfo['MerchantId'],
            'RqHeader' => [
                'Timestamp' => time(),
                'Revision' => '3.0.0',
            ],
            'Data' => $data,
        ];

        $response = $postService->post($input, $apiUrl);

        $this->_loggerInterface->debug('checkBarcode input:' . print_r($input, true));
        $this->_loggerInterface->debug('checkBarcode response:' . print_r($response, true));

        // 呼叫財政部API失敗
        if (isset($response['Data']['RtnCode']) && $response['Data']['RtnCode'] == 9000001) {
            $this->_loggerInterface->debug(__('code_1901'));
            throw new Exception(__('code_1901'), '1901');
        }

        // 手機條碼驗證失敗
        if (!isset($response['Data']['RtnCode']) || $response['Data']['RtnCode'] != 1 || $response['Data']['IsExist'] == 'N') {
            throw new Exception(__('code_1009'), '1009');
        }

        // 轉為JSON格式
        $responseArray = [
            'code'  => '0999',
            'msg'   => __('code_0999'),
            'data'  => '',
        ];

        return json_encode($responseArray);
    }
   
    /**
     * {@inheritdoc}
     */
    public function checkLoveCode($loveCode)
    {
        // 捐贈碼格式驗證
        if (!preg_match('/^([xX]{1}[0-9]{2,6}|[0-9]{3,7})$/', $loveCode)) {
            throw new Exception(__('code_1010'), '1010');
        }

        // 取出是否為測試模式
        $invoiceStage = $this->_mainService->getInvoiceConfig('enabled_invoice_stage');
        $this->_loggerInterface->debug('invalidInvoice invoiceStage:' . print_r($invoiceStage, true));

        // 取得發票會員資訊
        $accountInfo = $this->getInvoiceAccountInfo($invoiceStage);

        $factory = new Factory([
            'hashKey'   => $accountInfo['HashKey'],
            'hashIv'    => $accountInfo['HashIv'],
        ]);

        $postService = $factory->create('PostWithAesJsonResponseService');

        // 取出 URL
        $apiUrl = $this->_invoiceService->getApiUrl('check_love_code', $invoiceStage);
        $this->_loggerInterface->debug('checkLoveCode apiUrl:'. print_r($apiUrl, true));

        // 組合送綠界格式
        $data = [
            'MerchantID' 	=> $accountInfo['MerchantId'],
            'LoveCode' 		=> $loveCode,
        ];

        $input = [
            'MerchantID' => $accountInfo['MerchantId'],
            'RqHeader' => [
                'Timestamp' => time(),
                'Revision' => '3.0.0',
            ],
            'Data' => $data,
        ];

        $response = $postService->post($input, $apiUrl);

        $this->_loggerInterface->debug('checkLoveCode input:' . print_r($input, true));
        $this->_loggerInterface->debug('checkLoveCode response:' . print_r($response, true));

        // 呼叫財政部API失敗
        if (isset($response['Data']['RtnCode']) && $response['Data']['RtnCode'] == 9000001) {
            $this->_loggerInterface->debug(__('code_1901'));
            throw new Exception(__('code_1901'), '1901');
        }

        // 捐贈碼驗證失敗
        if (!isset($response['Data']['RtnCode']) || $response['Data']['RtnCode'] != 1 || $response['Data']['IsExist'] == 'N') {
            throw new Exception(__('code_1011'), '1011');
        }

        $responseArray = [
            'code'  => '0999',
            'msg'   => __('code_0999'),
            'data'  => '',
        ];

        return json_encode($responseArray);
    }

    /**
     * {@inheritdoc}
     */
    public function checkCitizenDigitalCertificate($carrierNumber)
    {
        // 自然人憑證格式驗證
        if (!preg_match('/^[a-zA-Z]{2}\d{14}$/', $carrierNumber)) {
            throw new Exception(__('code_1012'), '1012');
        }

        $responseArray = [
            'code'  => '0999',
            'msg'   => __('code_0999'),
            'data'  => '',
        ];

        return json_encode($responseArray);
    }

    /**
     * {@inheritdoc}
     */
    public function checkBusinessNumber($businessNumber)
    {
        // 統一編號格式驗證
        if (!preg_match('/^[0-9]{8}$/', $businessNumber)) {
            throw new Exception(__('code_1013'), '1013');
        }

        $responseArray = [
            'code'  => '0999',
            'msg'   => __('code_0999'),
            'data'  => '',
        ];

        return json_encode($responseArray);
    }

    /**
     * {@inheritdoc}
     */
    public function createInvoice($orderId, $protectCode)
    {
        $responseArray = [
            'code'  => '0000',
            'msg'   => '這是開立發票API',
            'data'  => '',
        ];
        
        // $enctyOrderId = str_replace(' ', '+', $orderId) ;
        // $orderId      = $this->_encryptionsService->decrypt($enctyOrderId);
        // $orderId      = (int) $orderId ;

        // 取出訂單protect_code資訊
        $protectCodeFromOrder = $this->_orderService->getProtectCode($orderId);
        $this->_loggerInterface->debug('createInvoice protectCodeFromOrder:'. print_r($protectCodeFromOrder,true));

        // 驗證訂單欄位protect_code 是否正確
        if ($protectCodeFromOrder != $protectCode) {
            $this->_loggerInterface->debug('createInvoice protect_code驗證錯誤:'. print_r($protectCode,true));

            // 轉為JSON格式
            $responseArray = [
                'code'  => '0001',
                'msg'   => __('code_0001'),
                'data'  => '',
            ];
        }
        else {
            //開立發票
            try{
                $responseArray = $this->_invoiceService->invoiceIssue($orderId);
            }catch(Exception $e){
                $this->_loggerInterface->debug('createInvoice Error Message: 開立發票失敗');
            }
        }
        $this->_loggerInterface->debug('createInvoice responseArray:'. print_r($responseArray,true));
        
        return json_encode($responseArray);
    }

    /**
     * {@inheritdoc}
     */
    public function getInvoiceTag($orderId, $protectCode)
    {
        $responseArray = [
            'code'  => '0000',
            'msg'   => '這是查詢發票開立標籤API',
            'data'  => '',
        ];

        // 解密訂單編號
        // $enctyOrderId = str_replace(' ', '+', $orderId) ;
        // $orderId      = $this->_encryptionsService->decrypt($enctyOrderId);
        // $orderId = (int) $orderId ;

        // $this->_loggerInterface->debug('getInvoiceTag enctyOrderId:'. print_r($enctyOrderId,true));
        $this->_loggerInterface->debug('getInvoiceTag orderId:'. print_r($orderId,true));

        // 取出訂單protect_code資訊
        $protectCodeFromOrder = $this->_orderService->getProtectCode($orderId);
        $this->_loggerInterface->debug('getInvoiceTag protectCodeFromOrder:'. print_r($protectCodeFromOrder,true));

        // 驗證訂單欄位protect_code 是否正確
        if($protectCodeFromOrder != $protectCode){

            $this->_loggerInterface->debug('getInvoiceTag protect_code驗證錯誤:'. print_r($protectCode,true));

            // 轉為JSON格式
            $responseArray = [
                'code'  => '0001',
                'msg'   => __('code_0001'),
                'data'  => '',
            ];

            return json_encode($responseArray);
        }

        // 取出發票開立標籤
        $sunpayInvoiceTag = $this->_orderService->getSunpayInvoiceTag($orderId);
        $this->_loggerInterface->debug('getInvoiceTag sunpayInvoiceTag:'. print_r($sunpayInvoiceTag,true));

        // 轉為JSON格式
        $responseArray = [
            'code'  => '0999',
            'msg'   => __('code_0999'),
            'data'  => $sunpayInvoiceTag,
        ];

        return json_encode($responseArray);
    }

    /**
     * {@inheritdoc}
     */
    public function getInvoiceMainConfig()
    {
        $responseArray = [
            'code'  => '0000',
            'msg'   => '這是查詢發票模組啟用API',
            'data'  => '',
        ];

        // 判斷發票模組是否啟動
        $sunpayEnableInvoice = $this->_mainService->isInvoiceModuleEnable() ;
        $this->_loggerInterface->debug('createInvoice sunpayEnableInvoice:'. print_r($sunpayEnableInvoice,true));

        // 轉為JSON格式
        $responseArray = [
            'code'  => '0999',
            'msg'   => __('code_0999'),
            'data'  => $sunpayEnableInvoice,
        ];

        // 轉為JSON格式
        return json_encode($responseArray);
    }

    /**
     * @return array $accountInfo
     */
    protected function getInvoiceAccountInfo()
    {

        // 取出 KEY IV MID (正式模式)
        $invoiceCompanyId  = $this->_mainService->getInvoiceConfig('ei_company_id');
        $invoiceMerchantId = $this->_mainService->getInvoiceConfig('ei_merchant_id');
        $invoiceHashKey    = $this->_mainService->getInvoiceConfig('ei_hashkey');
        $invoiceHashIv     = $this->_mainService->getInvoiceConfig('ei_hashiv');

        $this->_loggerInterface->debug('getInvoiceAccountInfo invoiceCompanyId:' . print_r($invoiceCompanyId, true));
        $this->_loggerInterface->debug('getInvoiceAccountInfo invoiceMerchantId:' . print_r($invoiceMerchantId, true));
        $this->_loggerInterface->debug('getInvoiceAccountInfo invoiceHashKey:' . print_r($invoiceHashKey, true));
        $this->_loggerInterface->debug('getInvoiceAccountInfo invoiceHashIv:' . print_r($invoiceHashIv, true));

        $accountInfo = [
            'CompanyId'  => $invoiceCompanyId,
            'MerchantId' => $invoiceMerchantId,
            'HashKey'    => $invoiceHashKey,
            'HashIv'     => $invoiceHashIv,
        ];

        return $accountInfo;
    }
}